/**
* @NApiVersion 2.1
* @NScriptType Suitelet
* @NModuleScope Public
*/

/* 

------------------------------------------------------------------------------------------
Script Information
------------------------------------------------------------------------------------------

Name:
SuiteQL Query Tool (Enhanced)

ID:
_suiteql_query_tool_enhanced

Description
A utility for running SuiteQL queries in a NetSuite instance.
Enhanced with syntax highlighting, auto-formatting, and optimized large data display.


------------------------------------------------------------------------------------------
MIT License
------------------------------------------------------------------------------------------

Copyright (c) 2021 Timothy Dietrich.
Enhanced version (c) 2024.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.


------------------------------------------------------------------------------------------
Developer(s)
------------------------------------------------------------------------------------------

Tim Dietrich
* timdietrich@me.com
* https://timdietrich.me

Enhanced Version Features:
* SQL Syntax Highlighting using CodeMirror
* Auto-formatting SQL queries
* Virtual scrolling for large datasets
* Improved performance with deferred rendering
* Professional UX/UI based on SQL IDE best practices

*/

var
	datatablesEnabled = true,
	remoteLibraryEnabled = true,
	rowsReturnedDefault = 25,
	queryFolderID = null,
	toolUpgradesEnabled = true,
	workbooksEnabled = false,
	virtualScrollPageSize = 100;

// ============================================
// LOCAL LIBRARY PATH CONFIGURATION
// Set the internal ID of the folder containing your library files
// You can find this ID in the File Cabinet by clicking on the folder
// ============================================
var libFolderID = null;  // Set to your library folder's internal ID (e.g., 12345)
var libFiles = {};  // Will be populated with filename -> url mapping

var 	
	file,
	https,
	log,
	page,
	query,
	record,
	render,
	runtime,	
	scriptURL,
	url,
	version = '2024.1-Enhanced';


define( [ 'N/file', 'N/https', 'N/log', 'N/ui/message', 'N/query', 'N/record', 'N/render', 'N/runtime', 'N/ui/serverWidget', 'N/url' ], main );


function main( fileModule, httpsModule, logModule, messageModule, queryModule, recordModule, renderModule, runtimeModule, serverWidgetModule, urlModule ) {

	file = fileModule;
	https = httpsModule;
	log = logModule;
	message = messageModule;
	query= queryModule;
	record = recordModule;
	render = renderModule;
	runtime = runtimeModule;
	serverWidget = serverWidgetModule;	
	url = urlModule;
	
    return {
    
    	onRequest: function( context ) {

			scriptURL = url.resolveScript( { scriptId: runtime.getCurrentScript().id, deploymentId: runtime.getCurrentScript().deploymentId, returnExternalURL: false } );

			// Load library file URLs from File table
			loadLibraryFiles();

    		if ( context.request.method == 'POST' ) {
    			postRequestHandle( context );
    		} else {
    			getRequestHandle( context );
			}
    		    			
        }
        
    }

}


// ============================================
// LIBRARY FILES LOADER
// Queries the File table to get URLs for library files
// ============================================
function loadLibraryFiles() {

	if ( libFolderID === null ) {
		log.error( { title: 'Library Configuration Error', details: 'libFolderID is not set. Please set the folder ID for your library files.' } );
		return;
	}

	try {

		var sql = `
			SELECT
				name,
				url
			FROM
				File
			WHERE
				folder = ?
		`;

		var queryResults = query.runSuiteQL( { query: sql, params: [ libFolderID ] } ).asMappedResults();

		// Create a mapping of filename -> url
		for ( var i = 0; i < queryResults.length; i++ ) {
			var fileName = queryResults[i].name;
			var fileUrl = queryResults[i].url;
			libFiles[ fileName ] = fileUrl;
		}

		log.debug( { title: 'Library Files Loaded', details: 'Loaded ' + queryResults.length + ' files from folder ID ' + libFolderID } );

	} catch( e ) {
		log.error( { title: 'loadLibraryFiles Error', details: e } );
	}

}


// Helper function to get library file URL
function getLibUrl( fileName ) {
	if ( libFiles[ fileName ] ) {
		return libFiles[ fileName ];
	}
	// Fallback - return empty string if file not found
	log.audit( { title: 'Library File Not Found', details: 'File not found in libFiles: ' + fileName } );
	return '';
}


function documentGenerate( context ) {

	try {

		var sessionScope = runtime.getCurrentSession();
		
		var docInfo = JSON.parse( sessionScope.get( { name: 'suiteQLDocumentInfo' } ) );
						
		var moreRecords = true;	
		
		var paginatedRowBegin = docInfo.rowBegin;
		
		var paginatedRowEnd = docInfo.rowEnd;		
		
		var queryParams = new Array();

		var records = new Array();

		do {			
	
			var paginatedSQL = 'SELECT * FROM ( SELECT ROWNUM AS ROWNUMBER, * FROM (' + docInfo.query + ' ) ) WHERE ( ROWNUMBER BETWEEN ' + paginatedRowBegin + ' AND ' + paginatedRowEnd + ')';
		
			var queryResults = query.runSuiteQL( { query: paginatedSQL, params: queryParams } ).asMappedResults(); 	
				
			records = records.concat( queryResults );	
					
			if ( queryResults.length < 5000 ) { moreRecords = false; }
		
			paginatedRowBegin = paginatedRowBegin + 5000;
				
		} while ( moreRecords );	
				
		var recordsDataSource = { 'records': records };	

		var renderer = render.create();
		renderer.addCustomDataSource( { alias: 'results', format: render.DataSource.OBJECT, data: recordsDataSource } );										
		renderer.templateContent = docInfo.template;
		
		if ( docInfo.docType == 'pdf' ) {
			let renderObj = renderer.renderAsPdf();				
			let pdfString = renderObj.getContents();						
			context.response.setHeader( 'Content-Type', 'application/pdf' );										
			context.response.write( pdfString );
		} else {
			let htmlString = renderer.renderAsString();							
			context.response.setHeader( 'Content-Type', 'text/html' );										
			context.response.write( htmlString );		
		}	
								
	} catch( e ) {		

		log.error( { title: 'documentGenerate Error', details: e } );
		
		context.response.write( 'Error: ' + e );		
		
	}				
	
}


function documentSubmit( context, requestPayload ) {

	try {		
	
		var responsePayload;		
		
		var sessionScope = runtime.getCurrentSession();
		
		sessionScope.set( { name: 'suiteQLDocumentInfo', value: JSON.stringify( requestPayload ) } );		
				
		responsePayload = { 'submitted': true }

	} catch( e ) {		

		log.error( { title: 'queryExecute Error', details: e } );
		
		responsePayload = { 'error': e }		
		
	}			
	
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );	
	
}


function getRequestHandle( context ) {
				
	if ( context.request.parameters.hasOwnProperty( 'function' ) ) {	
	
		if ( context.request.parameters['function'] == 'tablesReference' ) { htmlGenerateTablesReference( context ); }
	
		if ( context.request.parameters['function'] == 'documentGenerate' ) { documentGenerate( context ); }				
	
	} else {
															
		var form = serverWidget.createForm( { title: `SuiteQL Query Tool`, hideNavBar: false } );		
		
		var htmlField = form.addField(
			{
				id: 'custpage_field_html',
				type: serverWidget.FieldType.INLINEHTML,
				label: 'HTML'
			}								
		);

		htmlField.defaultValue = htmlGenerateTool();						

		context.response.writePage( form );					
		
	}
				
}


function htmlDataTablesFormatOption() {

	if ( datatablesEnabled === true ) {
	
		return `
			<div class="form-check-inline">
				<label class="form-check-label">
					<input type="radio" class="form-check-input" name="resultsFormat" value="datatable" onChange="responseGenerate();">DataTable
				</label>
			</div>			
 		`
	
	} else {
	
		return ``
	
	}

}


function htmlEnableViewsOption() {

	if ( queryFolderID !== null ) {
	
		return `
			<div class="options-section">
				<div class="form-check">
					<label class="form-check-label">
						<input type="checkbox" class="form-check-input" id="enableViews" checked>Enable Virtual Views
					</label>
				</div>																									
			</div>	
		`;
		
	} else {
		return ``
	}

}	


function htmlGenerateTablesReference( context ) {

	var form = serverWidget.createForm( { title: 'SuiteQL Tables Reference', hideNavBar: false } );

	var htmlField = form.addField(
		{
			id: 'custpage_field_html',
			type: serverWidget.FieldType.INLINEHTML,
			label: 'HTML'
		}								
	);

	htmlField.defaultValue = `

		<link rel="stylesheet" href="${getLibUrl('bootstrap.min.css')}">
		<script src="/ui/jquery/jquery-3.5.1.min.js"></script>
		<script src="${getLibUrl('bootstrap.min.js')}"></script>
		${jsFunctionDataTablesExternals()}		
		
		<style type = "text/css"> 
			body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif; }
			
			input[type="text"], input[type="search"], textarea, button {
				outline: none;
				box-shadow:none !important;
				border: 1px solid #d1d5db !important;
			}
			
			p, pre { font-size: 13px; }
			
			td, th { 
				font-size: 13px;
				border: 1px solid #e5e7eb;
				padding: 8px 12px;
			}
			
			th {
				font-weight: 600;
				background: #f9fafb;
			}
			
		</style>		
		
		<table style="table-layout: fixed; width: 100%; border-spacing: 6px; border-collapse: separate;">	
			<tr>
				<td width="30%" valign="top">
					<p style="color: #374151; font-weight: 600;">Select a table to view its details.</p>
					<div style="margin-top: 3px;" id="tablesColumn">Loading Tables Index...</div>
				</td>
				<td id="tableInfoColumn" valign="top">&nbsp;</td>			
			</tr>
		</table>
		
		<script>	
							
			window.jQuery = window.$ = jQuery;			
			
			${jsFunctionTableDetailsGet()}
			${jsFunctionTableNamesGet()}
			${jsFunctionTableQueryCopy()}						
			
			tableNamesGet();			
			
		</script>
		
	`;	

	context.response.writePage( form );		

}


function htmlGenerateTool() {

	return `

		<link rel="stylesheet" href="${getLibUrl('bootstrap.min.css')}">
		<script src="/ui/jquery/jquery-3.5.1.min.js"></script>
		<script src="${getLibUrl('bootstrap.min.js')}"></script>

		<!-- Google Fonts - JetBrains Mono & Fira Code -->
		<link rel="preconnect" href="https://fonts.googleapis.com">
		<link rel="preconnect" href="https://fonts.gstatic.com" crossorigin>
		<link href="https://fonts.googleapis.com/css2?family=Fira+Code:wght@400;500&family=JetBrains+Mono:wght@400;500&family=Source+Code+Pro:wght@400;500&display=swap" rel="stylesheet">

		<!-- CodeMirror for Syntax Highlighting -->
		<link rel="stylesheet" href="${getLibUrl('codemirror.min.css')}">
		<link rel="stylesheet" href="${getLibUrl('show-hint.min.css')}">
		<script src="${getLibUrl('codemirror.min.js')}"></script>
		<script src="${getLibUrl('sql.min.js')}"></script>
		<script src="${getLibUrl('show-hint.min.js')}"></script>
		<script src="${getLibUrl('sql-hint.min.js')}"></script>
		<script src="${getLibUrl('matchbrackets.min.js')}"></script>
		<script src="${getLibUrl('closebrackets.min.js')}"></script>
		<script src="${getLibUrl('active-line.min.js')}"></script>

		${jsFunctionDataTablesExternals()}
		
		<style type="text/css"> 
		
			/* ============================================
			   BASE STYLES
			   ============================================ */
			   
			:root {
				--font-ui: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, 'Helvetica Neue', Arial, sans-serif;
				--font-code: 'JetBrains Mono', 'Fira Code', 'Source Code Pro', 'Consolas', 'Monaco', 'Courier New', monospace;
				--color-primary: #3b82f6;
				--color-primary-dark: #2563eb;
				--color-success: #10b981;
				--color-success-dark: #059669;
				--color-gray-50: #f9fafb;
				--color-gray-100: #f3f4f6;
				--color-gray-200: #e5e7eb;
				--color-gray-300: #d1d5db;
				--color-gray-400: #9ca3af;
				--color-gray-500: #6b7280;
				--color-gray-600: #4b5563;
				--color-gray-700: #374151;
				--color-gray-800: #1f2937;
				--color-gray-900: #111827;
				--border-radius: 6px;
				--shadow-sm: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
				--shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px -1px rgba(0, 0, 0, 0.1);
			}
			
			body {
				font-family: var(--font-ui);
				font-size: 14px;
				color: var(--color-gray-700);
				line-height: 1.5;
				overflow-x: hidden;
			}

			#queryUI {
				max-width: 100vw;
				overflow: hidden;
				box-sizing: border-box;
			}

			#resultsDiv {
				max-width: 100%;
				overflow: hidden;
				contain: inline-size;
			}

			/* Force all children to respect container */
			#resultsDiv > * {
				max-width: 100%;
				box-sizing: border-box;
			}
		
			input[type="text"], input[type="search"], input[type="number"], textarea, button {
				outline: none;
				box-shadow: none !important;
				border: 1px solid var(--color-gray-300) !important;
				font-size: 13px;
			}
			
			input[type="number"] {
				padding: 4px 8px;
			}
			
			p, pre {
				font-size: 13px;
				margin-bottom: 8px;
			}
			
			/* ============================================
			   CODEMIRROR - SQL EDITOR STYLING
			   ============================================ */
			   
			#queryEditorContainer {
				height: 400px;
			}

			#queryEditorContainer .CodeMirror {
				height: 100%;
			}

			.CodeMirror {
				font-family: var(--font-code);
				font-size: 13px;
				line-height: 1.5;
				height: 400px;
				border: 1px solid var(--color-gray-300);
				border-radius: var(--border-radius);
				background: #fff;
			}

			.CodeMirror-scroll {
				height: 100%;
			}
			
			.CodeMirror-gutters {
				background: var(--color-gray-50);
				border-right: 1px solid var(--color-gray-200);
			}
			
			.CodeMirror-linenumber {
				color: var(--color-gray-400);
				font-size: 12px;
				padding: 0 8px 0 4px;
			}
			
			.CodeMirror-activeline-background {
				background: rgba(59, 130, 246, 0.05);
			}
			
			.CodeMirror-focused .CodeMirror-activeline-background {
				background: rgba(59, 130, 246, 0.08);
			}
			
			/* SQL Syntax Highlighting - Light Theme */
			.CodeMirror .cm-keyword { 
				color: #7c3aed; 
				font-weight: 500; 
			}
			.CodeMirror .cm-string { 
				color: #059669; 
			}
			.CodeMirror .cm-string-2 { 
				color: #0891b2; 
			}
			.CodeMirror .cm-number { 
				color: #0284c7; 
			}
			.CodeMirror .cm-comment { 
				color: var(--color-gray-400); 
				font-style: italic; 
			}
			.CodeMirror .cm-operator { 
				color: var(--color-gray-700); 
			}
			.CodeMirror .cm-builtin { 
				color: #0891b2; 
				font-weight: 500; 
			}
			.CodeMirror .cm-variable { 
				color: var(--color-gray-800); 
			}
			.CodeMirror .cm-variable-2 { 
				color: #b45309; 
			}
			.CodeMirror .cm-def { 
				color: #0f766e; 
			}
			.CodeMirror .cm-bracket { 
				color: var(--color-gray-600); 
			}
			.CodeMirror .cm-atom { 
				color: #7c3aed; 
			}
			
			/* Dark Theme */
			.cm-dark .CodeMirror {
				background: #1e1e2e;
				color: #cdd6f4;
				border-color: #313244;
			}
			.cm-dark .CodeMirror-gutters {
				background: #181825;
				border-color: #313244;
			}
			.cm-dark .CodeMirror-linenumber {
				color: #6c7086;
			}
			.cm-dark .CodeMirror-activeline-background {
				background: rgba(137, 180, 250, 0.1);
			}
			.cm-dark .CodeMirror .cm-keyword { color: #cba6f7; }
			.cm-dark .CodeMirror .cm-string { color: #a6e3a1; }
			.cm-dark .CodeMirror .cm-number { color: #fab387; }
			.cm-dark .CodeMirror .cm-comment { color: #6c7086; }
			.cm-dark .CodeMirror .cm-operator { color: #89dceb; }
			.cm-dark .CodeMirror .cm-builtin { color: #89b4fa; }
			.cm-dark .CodeMirror .cm-variable { color: #cdd6f4; }
			.cm-dark .CodeMirror .cm-bracket { color: #9399b2; }
			
			.CodeMirror-hints {
				font-family: var(--font-code);
				font-size: 12px;
				border: 1px solid var(--color-gray-200);
				border-radius: var(--border-radius);
				box-shadow: var(--shadow);
			}
			
			.CodeMirror-hints {
				font-family: var(--font-code);
				font-size: 12px;
				max-height: 250px;
				overflow-y: auto;
				border: 1px solid var(--color-gray-300);
				border-radius: 4px;
				box-shadow: 0 4px 12px rgba(0,0,0,0.15);
				background: #ffffff;
				padding: 4px 0;
				z-index: 1000;
			}

			.CodeMirror-hint {
				padding: 3px 10px;
				color: var(--color-gray-700);
				background: #ffffff;
				cursor: pointer;
				white-space: nowrap;
			}

			.CodeMirror-hint:hover {
				background: #f3f4f6;
			}

			li.CodeMirror-hint-active {
				background: var(--color-primary);
				color: white;
			}

			/* Dark theme hints */
			.cm-dark .CodeMirror-hints {
				background: #1e1e2e;
				border-color: #45475a;
			}

			.cm-dark .CodeMirror-hint {
				color: #cdd6f4;
				background: #1e1e2e;
			}

			.cm-dark .CodeMirror-hint:hover {
				background: #313244;
			}

			.cm-dark li.CodeMirror-hint-active {
				background: var(--color-primary);
				color: white;
			}
			
			/* ============================================
			   BUTTONS
			   ============================================ */
			   
			.btn {
				font-size: 13px;
				font-weight: 500;
				padding: 6px 12px;
				border-radius: var(--border-radius);
				transition: all 0.15s ease;
			}
			
			.btn-sm {
				padding: 5px 10px;
				font-size: 12px;
			}
			
			.btn-light {
				background: white;
				border-color: var(--color-gray-300) !important;
				color: var(--color-gray-700);
			}
			
			.btn-light:hover {
				background: var(--color-gray-50);
				border-color: var(--color-gray-400) !important;
			}
			
			.btn-success {
				background: var(--color-success);
				border-color: var(--color-success) !important;
				color: white;
			}
			
			.btn-success:hover {
				background: var(--color-success-dark);
				border-color: var(--color-success-dark) !important;
			}
			
			.btn-primary {
				background: var(--color-primary);
				border-color: var(--color-primary) !important;
				color: white;
			}
			
			.btn-primary:hover {
				background: var(--color-primary-dark);
				border-color: var(--color-primary-dark) !important;
			}
			
			.btn-format {
				background: var(--color-gray-100);
				color: var(--color-gray-700);
				border: 1px solid var(--color-gray-300) !important;
			}

			.btn-format:hover {
				background: var(--color-gray-200);
				color: var(--color-gray-800);
			}
			
			/* ============================================
			   OPTIONS PANEL
			   ============================================ */
			   
			.options-panel {
				background: var(--color-gray-50);
				border: 1px solid var(--color-gray-200);
				border-radius: var(--border-radius);
				padding: 16px;
				height: 400px;
				overflow-y: auto;
				box-sizing: border-box;
			}
			
			.options-section {
				padding: 12px 0;
				border-bottom: 1px solid var(--color-gray-200);
			}
			
			.options-section:first-child {
				padding-top: 0;
			}
			
			.options-section:last-child {
				border-bottom: none;
				padding-bottom: 0;
			}
			
			.options-section p {
				font-size: 12px;
				font-weight: 600;
				color: var(--color-gray-600);
				text-transform: uppercase;
				letter-spacing: 0.025em;
				margin-bottom: 8px;
			}
			
			.form-check-label {
				font-size: 13px;
				color: var(--color-gray-700);
				cursor: pointer;
			}
			
			.form-check-input {
				margin-right: 6px;
			}
			
			.form-check-inline {
				margin-right: 12px;
			}
			
			/* ============================================
			   TABLE STYLES
			   ============================================ */

			.table-responsive {
				position: relative;
				width: 100%;
				overflow: auto;
				border: 1px solid var(--color-gray-200);
				border-radius: var(--border-radius);
				max-height: 500px;
			}

			.results-table {
				font-size: 11px;
				border-collapse: collapse;
				white-space: nowrap;
			}

			.results-table th {
				position: sticky;
				top: 0;
				z-index: 1;
				font-size: 11px;
				font-weight: 600;
				text-transform: lowercase;
				background: var(--color-gray-100);
				color: var(--color-gray-600);
				padding: 4px 8px;
				border-bottom: 1px solid var(--color-gray-300);
				border-right: 1px solid var(--color-gray-200);
				cursor: pointer;
				user-select: none;
				position: relative;
			}

			.results-table th:hover {
				background: var(--color-gray-200);
			}

			.results-table th.sortable::after {
				content: '\\2195';
				font-size: 10px;
				color: var(--color-gray-400);
				margin-left: 4px;
			}

			.results-table th.sort-asc::after {
				content: '\\25B2';
				color: var(--color-gray-600);
			}

			.results-table th.sort-desc::after {
				content: '\\25BC';
				color: var(--color-gray-600);
			}

			.results-table th .resizer {
				position: absolute;
				right: 0;
				top: 0;
				height: 100%;
				width: 5px;
				cursor: col-resize;
				background: transparent;
			}

			.results-table th .resizer:hover,
			.results-table th .resizer.resizing {
				background: var(--color-primary);
			}

			.results-table th:last-child {
				border-right: none;
			}

			.results-table td {
				padding: 2px 8px;
				border-bottom: 1px solid var(--color-gray-100);
				border-right: 1px solid var(--color-gray-100);
				font-family: var(--font-code);
				font-size: 11px;
				line-height: 1.4;
				max-width: 300px;
				overflow: hidden;
				text-overflow: ellipsis;
			}

			.results-table td:last-child {
				border-right: none;
			}

			.results-table tbody tr:hover td {
				background: rgba(59, 130, 246, 0.06);
			}

			.results-table tbody tr:last-child td {
				border-bottom: none;
			}

			/* ============================================
			   STATS BAR
			   ============================================ */

			.stats-bar {
				background: transparent;
				padding: 0 0 6px 0;
				margin: 0;
				display: inline-flex;
				gap: 12px;
				font-size: 11px;
			}

			.stat-item {
				display: inline-flex;
				align-items: center;
				gap: 3px;
				color: var(--color-gray-500);
			}

			.stat-value {
				font-weight: 600;
				color: var(--color-gray-700);
			}
			
			/* ============================================
			   VIRTUAL SCROLL
			   ============================================ */

			.virtual-scroll-container {
				max-height: 450px;
				overflow: auto;
				border: 1px solid var(--color-gray-200);
				border-radius: var(--border-radius);
				position: relative;
			}

			.virtual-scroll-table {
				border-collapse: collapse;
				white-space: nowrap;
			}

			.virtual-scroll-table th {
				position: sticky;
				top: 0;
				background: var(--color-gray-100);
				z-index: 1;
				font-size: 11px;
				font-weight: 600;
				text-transform: lowercase;
				color: var(--color-gray-600);
				padding: 4px 8px;
				border-bottom: 1px solid var(--color-gray-300);
				border-right: 1px solid var(--color-gray-200);
				cursor: pointer;
				user-select: none;
				position: relative;
			}

			.virtual-scroll-table th:hover {
				background: var(--color-gray-200);
			}

			.virtual-scroll-table th.sortable::after {
				content: '\\2195';
				font-size: 10px;
				color: var(--color-gray-400);
				margin-left: 4px;
			}

			.virtual-scroll-table th.sort-asc::after {
				content: '\\25B2';
				color: var(--color-gray-600);
			}

			.virtual-scroll-table th.sort-desc::after {
				content: '\\25BC';
				color: var(--color-gray-600);
			}

			.virtual-scroll-table th .resizer {
				position: absolute;
				right: 0;
				top: 0;
				height: 100%;
				width: 5px;
				cursor: col-resize;
				background: transparent;
			}

			.virtual-scroll-table th .resizer:hover,
			.virtual-scroll-table th .resizer.resizing {
				background: var(--color-primary);
			}

			.virtual-scroll-table th:last-child {
				border-right: none;
			}

			.virtual-scroll-table td {
				padding: 2px 8px;
				border-bottom: 1px solid var(--color-gray-100);
				border-right: 1px solid var(--color-gray-100);
				font-family: var(--font-code);
				font-size: 11px;
				line-height: 1.4;
				max-width: 300px;
				overflow: hidden;
				text-overflow: ellipsis;
			}

			.virtual-scroll-table td:last-child {
				border-right: none;
			}

			.virtual-scroll-table tbody tr:hover td {
				background: rgba(59, 130, 246, 0.06);
			}
			
			/* ============================================
			   PAGINATION
			   ============================================ */
			   
			.pagination-controls {
				display: flex;
				align-items: center;
				gap: 8px;
				margin: 12px 0;
			}
			
			.page-info {
				font-size: 13px;
				color: var(--color-gray-600);
				margin: 0 8px;
			}
			
			/* ============================================
			   LOADING & MISC
			   ============================================ */
			   
			.loading-overlay {
				padding: 40px;
				text-align: center;
				color: var(--color-gray-500);
			}
			
			.spinner {
				border: 3px solid var(--color-gray-200);
				border-top: 3px solid var(--color-primary);
				border-radius: 50%;
				width: 32px;
				height: 32px;
				animation: spin 0.8s linear infinite;
				margin: 0 auto 12px;
			}
			
			@keyframes spin {
				0% { transform: rotate(0deg); }
				100% { transform: rotate(360deg); }
			}

			/* ============================================
			   POPUP PROGRESS BAR (Non-blocking)
			   ============================================ */

			.query-progress-popup {
				position: fixed;
				bottom: 20px;
				right: 20px;
				background: var(--color-white);
				border: 1px solid var(--color-gray-200);
				border-radius: var(--border-radius);
				padding: 12px 16px;
				box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
				z-index: 9999;
				display: flex;
				align-items: center;
				gap: 12px;
				min-width: 200px;
				animation: slideIn 0.3s ease-out;
			}

			@keyframes slideIn {
				from {
					transform: translateX(100%);
					opacity: 0;
				}
				to {
					transform: translateX(0);
					opacity: 1;
				}
			}

			.query-progress-popup.hiding {
				animation: slideOut 0.3s ease-in forwards;
			}

			@keyframes slideOut {
				from {
					transform: translateX(0);
					opacity: 1;
				}
				to {
					transform: translateX(100%);
					opacity: 0;
				}
			}

			.query-progress-popup .spinner-small {
				border: 2px solid var(--color-gray-200);
				border-top: 2px solid var(--color-primary);
				border-radius: 50%;
				width: 20px;
				height: 20px;
				animation: spin 0.8s linear infinite;
				flex-shrink: 0;
			}

			.query-progress-popup .progress-text {
				font-size: 13px;
				color: var(--color-gray-700);
			}

			.query-progress-popup .progress-bar-container {
				flex: 1;
				height: 4px;
				background: var(--color-gray-200);
				border-radius: 2px;
				overflow: hidden;
			}

			.query-progress-popup .progress-bar {
				height: 100%;
				background: linear-gradient(90deg, var(--color-primary), var(--color-primary-dark));
				border-radius: 2px;
				animation: progressPulse 1.5s ease-in-out infinite;
				width: 30%;
			}

			@keyframes progressPulse {
				0% { transform: translateX(-100%); }
				100% { transform: translateX(400%); }
			}

			/* ============================================
			   TABLE SEARCH
			   ============================================ */

			.table-toolbar {
				display: flex;
				align-items: center;
				justify-content: flex-start;
				gap: 16px;
				margin-bottom: 6px;
				flex-wrap: wrap;
			}

			.table-search-container {
				display: flex;
				align-items: center;
				gap: 8px;
			}

			.table-search-input {
				width: 180px;
				padding: 4px 8px;
				font-size: 11px;
				border: 1px solid var(--color-gray-300);
				border-radius: 4px;
				background: var(--color-white);
				color: var(--color-gray-800);
			}

			.table-search-input:focus {
				outline: none;
				border-color: var(--color-primary);
			}

			.table-search-input::placeholder {
				color: var(--color-gray-400);
			}

			.search-stats {
				font-size: 11px;
				color: var(--color-gray-500);
			}

			.column-filter-select {
				padding: 4px 6px;
				font-size: 11px;
				border: 1px solid var(--color-gray-300);
				border-radius: 4px;
				background: var(--color-white);
				color: var(--color-gray-800);
				min-width: 100px;
				max-width: 150px;
			}

			.column-filter-select:focus {
				outline: none;
				border-color: var(--color-primary);
			}

			.results-table tr.hidden-row {
				display: none;
			}

			.theme-toggle {
				background: transparent;
				border: 1px solid var(--color-gray-300) !important;
				padding: 4px 8px;
				border-radius: var(--border-radius);
				cursor: pointer;
				font-size: 14px;
				margin-left: 8px;
			}
			
			.theme-toggle:hover {
				background: var(--color-gray-100);
			}
			
			.file-info {
				font-size: 12px;
				color: var(--color-gray-500);
				margin-top: 6px;
			}
			
			.file-info .text-danger {
				color: #ef4444;
			}
			
			.section-title {
				font-size: 14px;
				font-weight: 600;
				color: var(--color-gray-700);
				margin: 0;
			}
			
			.header-row {
				display: flex;
				align-items: center;
				justify-content: space-between;
				margin-bottom: 12px;
			}
			
			.button-group {
				display: flex;
				gap: 6px;
				align-items: center;
			}
			
			.editor-container {
				display: flex;
				gap: 16px;
				align-items: stretch;
			}
			
			.editor-main {
				flex: 1;
				min-width: 0;
				position: relative;
			}

			/* Editor Resize Handle */
			.editor-resize-handle {
				height: 8px;
				background: var(--color-gray-100);
				border: 1px solid var(--color-gray-300);
				border-top: none;
				border-radius: 0 0 var(--border-radius) var(--border-radius);
				cursor: ns-resize;
				display: flex;
				align-items: center;
				justify-content: center;
				transition: background 0.15s ease;
			}

			.editor-resize-handle:hover {
				background: var(--color-gray-200);
			}

			.editor-resize-handle::before {
				content: '';
				width: 40px;
				height: 3px;
				background: var(--color-gray-400);
				border-radius: 2px;
			}

			.editor-resize-handle:hover::before {
				background: var(--color-gray-500);
			}
			
			.editor-sidebar {
				width: 280px;
				flex-shrink: 0;
			}
			
			/* DataTables overrides */
			.dataTables_wrapper {
				font-size: 13px;
			}
			
			.dataTables_filter input {
				font-size: 13px;
				padding: 4px 8px;
			}
			
			.dataTables_length select {
				font-size: 13px;
				padding: 4px 8px;
			}
			
			/* Modal improvements */
			.modal-header {
				background: var(--color-gray-50);
				border-bottom: 1px solid var(--color-gray-200);
			}
			
			.modal-title {
				font-size: 16px;
				font-weight: 600;
			}
			
			/* Footer */
			.footer-text {
				text-align: center;
				font-size: 12px;
				color: var(--color-gray-400);
				padding: 16px 0;
			}
			
			.footer-text a {
				color: var(--color-gray-500);
			}
			
		</style>
		
		${htmlLocalLoadModal()}
		
		${htmlRemoteLoadModal()}	

		${htmlSaveModal()}
		
		${htmlWorkbooksModal()}
			
		${htmlQueryUI()}

		<script>	
		
			var
				activeSQLFile = {},
				queryResponsePayload,
				fileLoadResponsePayload,
				sqlEditor,
				templateEditor,
				isDarkTheme = false,
				virtualScrollState = {
					currentPage: 0,
					pageSize: ${virtualScrollPageSize},
					totalRows: 0
				};
			
			window.jQuery = window.$ = jQuery;
			
			$('#queryUI').show();
			$('#templateHeaderRow').hide();
			$('#templateFormRow').hide();
			
			// Initialize CodeMirror
			$(document).ready(function() {
				initCodeMirror();
			});
						
			${jqueryKeydownHandler()}
			${jqueryModalHandlers()}
			${jsFunctionSchemaCache()}
			${jsFunctionInitCodeMirror()}
			${jsFunctionSQLFormatter()}
			${jsFunctionFormatQuery()}
			${jsFunctionToggleTheme()}
			${jsFunctionDefaultQuerySet()}
			${jsFunctionDocumentGenerate()}
			${jsFunctionEnablePaginationToggle()}
			${jsFunctionFileInfoRefresh()}
			${jsFunctionHideRowNumbersToggle()}
			${jsFunctionLocalLibraryFilesGet()}
			${jsFunctionLocalSQLFileLoad()}
			${jsFunctionLocalSQLFileSave()}
			${jsFunctionQueryFormRowToggle()}
			${jsFunctionQueryProgress()}
			${jsFunctionQuerySubmit()}
			${jsFunctionQueryTextAreaResize()}
			${jsFunctionRadioFieldValueGet()}
			${jsFunctionRemoteLibraryIndexGet()}	
			${jsFunctionRemoteSQLFileLoad()}
			${jsFunctionResponseDataCopy()}
			${jsFunctionResponseGenerate()}
			${jsFunctionResponseGenerateCSV()}
			${jsFunctionResponseGenerateJSON()}
			${jsFunctionResponseGenerateTable()}
			${jsFunctionTableSearch()}
			${jsFunctionTableSortAndResize()}
			${jsFunctionVirtualScrollTable()}
			${jsFunctionReturnAllToggle()}
			${jsFunctiontablesReferenceOpen()}
			${jsFunctionWorkbookLoad()}
			${jsFunctionWorkbooksListGet()}

		</script>	
		
	`
	
}


function htmlLocalLoadModal() {

	return `
		<div class="modal fade" id="localLoadModal">
			<div class="modal-dialog modal-lg">
				<div class="modal-content">

					<div class="modal-header">
						<h5 class="modal-title">Local Query Library</h5>
						<button type="button" class="close" data-dismiss="modal">&times;</button>
					</div>

					<div class="modal-body" id="localSQLFilesList">								
					</div>

				</div>
			</div>
		</div>	
	`;	

}


function htmlQueryUI() {

	return `

		<div class="collapse" id="queryUI" style="text-align: left; padding: 0 8px;">	
		
			<!-- Header Row -->
			<div class="header-row">
				<div style="display: flex; align-items: center; gap: 12px;">
					<h5 class="section-title">
						<a href="#" onClick="javascript:defaultQuerySet();" title="Click to load a sample query." style="color: inherit; text-decoration: none;">Query Editor</a>
					</h5>
					<button type="button" class="theme-toggle" onclick="toggleTheme();" title="Toggle Dark/Light Theme">🌓</button>
				</div>
				
				<div class="button-group">
					<button type="button" class="btn btn-sm btn-format" onclick="formatQuery();" title="Auto-format SQL (Ctrl+Shift+F)">Format SQL</button>
					<button type="button" class="btn btn-sm btn-light" onClick="javascript:tablesReferenceOpen();">Tables</button>
					${jsFunctionWorkbooksButton()}	
					${jsFunctionRemoteLibraryButton()}	
					${jsFunctionLocalLibraryButtons()}
					<button type="button" class="btn btn-sm btn-success" onclick="querySubmit();" accesskey="r">▶ Run</button>	
				</div>
				
				<button id="btnQueryFormRowToggle" type="button" class="btn btn-sm btn-light" onclick="queryFormRowToggle();">Hide Editor</button>
			</div>
			
			<!-- Editor Container -->
			<div class="editor-container" id="queryFormRow">
				<div class="editor-main">
					<textarea id="query" style="display: none;"></textarea>
					<div id="queryEditorContainer"></div>
					<div class="editor-resize-handle" id="editorResizeHandle" title="Drag to resize editor"></div>
					<div class="file-info" id="fileInfo"></div>
				</div>
				
				<div class="editor-sidebar">
					<div class="options-panel">
						
						<!-- Pagination Options -->
						<div class="options-section">
							<div class="form-check">
								<label class="form-check-label">
									<input type="checkbox" class="form-check-input" id="enablePagination" onChange="enablePaginationToggle();">Enable Pagination
								</label>
							</div>
							
							<div id="paginationOptions" style="display: none; margin-top: 10px;">
								<p>Return Rows:</p>
								<div style="display: flex; align-items: center; gap: 8px;">
									<input type="number" class="form-control-sm" id="rowBegin" style="width: 70px;" value="1">
									<span style="color: #6b7280;">to</span>
									<input type="number" class="form-control-sm" id="rowEnd" style="width: 70px;" value="${rowsReturnedDefault}">
								</div>
								
								<div class="form-check" style="margin-top: 8px;">
									<label class="form-check-label">
										<input type="checkbox" class="form-check-input" id="returnAll" onChange="returnAllToggle();">Return All Rows
									</label>
								</div>
								
								<div class="form-check">
									<label class="form-check-label">
										<input type="checkbox" class="form-check-input" id="returnTotals">Show Total Count
									</label>
								</div>
								
								<div class="form-check">
									<label class="form-check-label">
										<input type="checkbox" class="form-check-input" id="hideRowNumbers" checked onChange="hideRowNumbersToggle();">Hide Row Numbers
									</label>
								</div>
							</div>
						</div>
						
						${htmlEnableViewsOption()}
						
						<!-- Results Format -->
						<div class="options-section" id="resultsFormatSection">
							<p>Results Format:</p>
							<div class="form-check">
								<label class="form-check-label">
									<input type="radio" class="form-check-input" name="resultsFormat" value="table" checked onChange="responseGenerate();">Table
								</label>
							</div>
							<div class="form-check">
								<label class="form-check-label">
									<input type="radio" class="form-check-input" name="resultsFormat" value="virtual" onChange="responseGenerate();">Virtual Scroll
								</label>
							</div>
							${htmlDataTablesFormatOption()}
							<div class="form-check">
								<label class="form-check-label">
									<input type="radio" class="form-check-input" name="resultsFormat" value="csv" onChange="responseGenerate();">CSV
								</label>
							</div>
							<div class="form-check">
								<label class="form-check-label">
									<input type="radio" class="form-check-input" name="resultsFormat" value="json" onChange="responseGenerate();">JSON
								</label>
							</div>
						</div>
						
						<!-- NULL Display -->
						<div class="options-section" id="nullFormatDiv">
							<p>NULL Values:</p>
							<div class="form-check-inline">
								<label class="form-check-label">
									<input type="radio" class="form-check-input" name="nullFormat" value="dimmed" checked onChange="responseGenerate();">Dim
								</label>
							</div>
							<div class="form-check-inline">
								<label class="form-check-label">
									<input type="radio" class="form-check-input" name="nullFormat" value="blank" onChange="responseGenerate();">Blank
								</label>
							</div>
							<div class="form-check-inline">
								<label class="form-check-label">
									<input type="radio" class="form-check-input" name="nullFormat" value="null" onChange="responseGenerate();">null
								</label>
							</div>
						</div>
						
						<!-- Large Data Options -->
						<div class="options-section">
							<p>Large Data:</p>
							<div class="form-check">
								<label class="form-check-label">
									<input type="checkbox" class="form-check-input" id="useDeferredRendering" checked>Deferred Rendering
								</label>
							</div>
							<div style="display: flex; align-items: center; gap: 8px; margin-top: 8px;">
								<span style="font-size: 12px; color: #6b7280;">Page Size:</span>
								<select class="form-control-sm" id="virtualPageSize" style="width: 70px; font-size: 12px;">
									<option value="50">50</option>
									<option value="100" selected>100</option>
									<option value="250">250</option>
									<option value="500">500</option>
								</select>
							</div>
						</div>
						
					</div>
				</div>
			</div>
			
			<!-- Template Row (hidden by default) -->
			<div id="templateHeaderRow" style="display: none; margin-top: 16px;">
				<div class="header-row">
					<h5 class="section-title">Template Editor</h5>
					<div class="button-group">
						<button type="button" class="btn btn-sm btn-light" onClick="window.open('https://bfo.com/products/report/docs/userguide.pdf');">BFO Ref</button>
						<button type="button" class="btn btn-sm btn-light" onClick="window.open('https://freemarker.apache.org/docs/index.html');">FreeMarker</button>
						<button type="button" class="btn btn-sm btn-success" onclick="documentGenerate();" accesskey="g">Generate</button>
					</div>
				</div>
			</div>
			
			<div id="templateFormRow" style="display: none;">
				<textarea 
					class="form-control" 
					id="template" 
					style="font-family: var(--font-code); font-size: 13px; line-height: 1.5; padding: 12px;"
					rows="15" 
					placeholder="Enter your template here."
				></textarea>
			</div>
			
			<!-- Results -->
			<div id="resultsDiv" style="margin-top: 16px; display: none;"></div>
			
			<!-- Footer -->
			<div class="footer-text">
				SuiteQL Query Tool v${version} • 
				<a href="https://timdietrich.me/" target="_blank">Tim Dietrich</a> • 
				Enhanced Edition
			</div>
		
		</div>

	`;	

}


function htmlRemoteLoadModal() {

	return `
		<div class="modal fade" id="remoteLoadModal">
			<div class="modal-dialog modal-lg">
				<div class="modal-content">

					<div class="modal-header">
						<h5 class="modal-title">Remote Query Library</h5>
						<button type="button" class="close" data-dismiss="modal">&times;</button>
					</div>

					<div class="modal-body" id="remoteSQLFilesList">								
					</div>

				</div>
			</div>
		</div>	
	`;	

}


function htmlSaveModal() {

	return `
		<div class="modal fade" id="saveModal">
			<div class="modal-dialog modal-lg">
				<div class="modal-content">

					<div class="modal-header">
						<h5 class="modal-title">Save Query</h5>
						<button type="button" class="close" data-dismiss="modal">&times;</button>
					</div>
					
					<div class="modal-body" id="saveQueryMessage" style="display: none;"></div>

					<div class="modal-body" id="saveQueryForm" style="display: none;">
						<div style="margin-bottom: 16px;">
							<label style="font-size: 13px; font-weight: 500; display: block; margin-bottom: 4px;">File Name</label>
							<input type="text" class="form-control" id="saveQueryFormFileName" style="max-width: 300px;">
						</div>
						<div style="margin-bottom: 16px;">
							<label style="font-size: 13px; font-weight: 500; display: block; margin-bottom: 4px;">Description</label>
							<input type="text" class="form-control" id="saveQueryFormDescription" style="max-width: 500px;">
						</div>
						<button type="button" class="btn btn-success" onclick="javascript:localSQLFileSave();">Save Query</button>
					</div>

				</div>
			</div>
		</div>
	`;	

}


function htmlWorkbooksModal() {

	return `
		<div class="modal fade" id="workbooksModal">
			<div class="modal-dialog modal-lg">
				<div class="modal-content">

					<div class="modal-header">
						<h5 class="modal-title">Workbooks</h5>
						<button type="button" class="close" data-dismiss="modal">&times;</button>
					</div>

					<div class="modal-body" id="workbooksList">								
					</div>

				</div>
			</div>
		</div>	
	`;	

}


function jqueryKeydownHandler() {

	return `
		$(document).keydown(function(e) {
			// Ctrl+Shift+F for format
			if (e.ctrlKey && e.shiftKey && e.keyCode === 70) {
				e.preventDefault();
				formatQuery();
				return false;
			}
			// Ctrl+Enter to run query
			if (e.ctrlKey && e.keyCode === 13) {
				e.preventDefault();
				querySubmit();
				return false;
			}
		});
	`
}


function jqueryModalHandlers() {

	return `
	
		$('#localLoadModal').on('shown.bs.modal', function(e) {
			localLibraryFilesGet();
		});
		
		$('#remoteLoadModal').on('shown.bs.modal', function(e) {
			remoteLibraryIndexGet();
		});
		
		$('#saveModal').on('shown.bs.modal', function(e) {
			document.getElementById('saveQueryMessage').style.display = "none";
			document.getElementById('saveQueryForm').style.display = "none";
			
			var queryValue = sqlEditor ? sqlEditor.getValue() : document.getElementById('query').value;
			
			if (queryValue == '') { 
				document.getElementById('saveQueryMessage').innerHTML = '<p style="color: #ef4444;">Please enter a query first.</p>';	
				document.getElementById('saveQueryMessage').style.display = "block";							
				return; 
			} else {
				document.getElementById('saveQueryForm').style.display = "block";
				
				if (activeSQLFile.hasOwnProperty('fileName')) {								
					document.getElementById('saveQueryFormFileName').value = activeSQLFile.fileName;									
				}
				
				if (activeSQLFile.hasOwnProperty('description')) {								
					document.getElementById('saveQueryFormDescription').value = activeSQLFile.description;									
				}								
				
				document.getElementById('saveQueryFormFileName').focus();
			}					
		});
		
		$('#workbooksModal').on('shown.bs.modal', function(e) {
			workbooksListGet();
		});
			
	`

}


function jsFunctionSchemaCache() {

	return `
		// Schema cache for autocomplete
		var schemaCache = {
			tables: {},
			tableList: [],
			aliases: {},
			ctes: {},
			loading: {},
			initialized: false
		};

		// Prefetch all table names in background
		function prefetchTableNames() {
			console.log('Schema: Starting prefetch...');

			var url = '/app/recordscatalog/rcendpoint.nl?action=getRecordTypes&data=' +
				encodeURI(JSON.stringify({ structureType: 'FLAT' }));

			var xhr = new XMLHttpRequest();
			xhr.open('GET', url, true);
			xhr.send();

			xhr.onload = function() {
				console.log('Schema: Got response, status=' + xhr.status);
				if (xhr.status === 200) {
					try {
						var response = JSON.parse(xhr.response);
						console.log('Schema: Parsed response', response);
						if (response.data && response.data.items) {
							schemaCache.tableList = response.data.items.map(function(item) {
								return item.id.toLowerCase();
							});
							schemaCache.initialized = true;
							console.log('Schema: Loaded ' + schemaCache.tableList.length + ' tables');
							console.log('Schema: Sample tables:', schemaCache.tableList.slice(0, 10));
						} else {
							console.log('Schema: No data.items in response');
						}
					} catch(e) {
						console.error('Schema: Failed to parse', e);
					}
				}
			};

			xhr.onerror = function() {
				console.error('Schema: XHR error');
			};
		}

		// Fetch columns for a specific table
		function fetchTableColumns(tableName, callback) {
			var tableKey = tableName.toLowerCase();

			if (schemaCache.tables[tableKey]) {
				if (callback) callback(schemaCache.tables[tableKey]);
				return;
			}

			// Mark as failed to prevent repeated attempts
			if (schemaCache.loading[tableKey] === 'failed') return;
			if (schemaCache.loading[tableKey]) return;
			schemaCache.loading[tableKey] = true;

			var url = '/app/recordscatalog/rcendpoint.nl?action=getRecordTypeDetail&data=' +
				encodeURI(JSON.stringify({ scriptId: tableName, detailType: 'SS_ANAL' }));

			var xhr = new XMLHttpRequest();
			xhr.open('GET', url, true);
			xhr.send();

			xhr.onload = function() {
				if (xhr.status === 200) {
					try {
						var response = JSON.parse(xhr.response);
						if (response.data && response.data.fields) {
							var columns = [];
							for (var i = 0; i < response.data.fields.length; i++) {
								var f = response.data.fields[i];
								if (f.isColumn) {
									columns.push(f.id.toLowerCase());
								}
							}
							schemaCache.tables[tableKey] = columns;
							schemaCache.loading[tableKey] = false;
							console.log('Schema: Loaded ' + columns.length + ' columns for ' + tableKey);
							if (callback) callback(columns);
						} else {
							schemaCache.loading[tableKey] = 'failed';
						}
					} catch(e) {
						schemaCache.loading[tableKey] = 'failed';
					}
				} else {
					// 500 or other error - mark as failed, don't retry
					schemaCache.loading[tableKey] = 'failed';
				}
			};

			xhr.onerror = function() {
				schemaCache.loading[tableKey] = 'failed';
			};
		}

		// Parse table aliases from query
		function parseTableAliases(query) {
			// Don't clear aliases - preserve CTE mappings from parseCTEs
			// Only clear table-related aliases
			var cteAliases = {};
			for (var key in schemaCache.aliases) {
				if (schemaCache.ctes[key] || schemaCache.ctes[schemaCache.aliases[key]]) {
					cteAliases[key] = schemaCache.aliases[key];
				}
			}
			schemaCache.aliases = cteAliases;

			var q = query.toLowerCase();

			// Reserved words
			var reserved = {
				'on':1, 'where':1, 'left':1, 'right':1, 'inner':1, 'outer':1,
				'cross':1, 'full':1, 'join':1, 'and':1, 'or':1, 'select':1,
				'from':1, 'group':1, 'order':1, 'having':1, 'limit':1, 'as':1,
				'case':1, 'when':1, 'then':1, 'else':1, 'end':1, 'not':1,
				'in':1, 'is':1, 'null':1, 'between':1, 'like':1, 'with':1
			};

			// Split by whitespace and punctuation
			var words = q.split(/[\\s,()]+/).filter(function(w) { return w.length > 0; });
			console.log('Schema: Parsing words:', words);

			for (var i = 0; i < words.length - 1; i++) {
				if (words[i] === 'from' || words[i] === 'join') {
					var tableName = words[i + 1];
					if (!tableName || reserved[tableName]) continue;

					var nextWord = words[i + 2];
					var alias = null;

					if (nextWord === 'as' && words[i + 3]) {
						alias = words[i + 3];
					} else if (nextWord && !reserved[nextWord]) {
						alias = nextWord;
					}

					console.log('Schema: Found table=' + tableName + ', alias=' + alias);

					// Check if this is a CTE reference
					if (schemaCache.ctes[tableName]) {
						console.log('Schema: ' + tableName + ' is a CTE');
						schemaCache.aliases[tableName] = tableName;
						if (alias && !reserved[alias]) {
							// Map alias to CTE name (not table name)
							schemaCache.aliases[alias] = tableName;
							console.log('Schema: Mapped CTE alias ' + alias + ' -> ' + tableName);
						}
						continue;
					}

					console.log('Schema: tableList has ' + schemaCache.tableList.length + ' items');
					console.log('Schema: Table in list?', schemaCache.tableList.indexOf(tableName) > -1);

					// Always map table and alias if table is in list
					if (schemaCache.tableList.length === 0 || schemaCache.tableList.indexOf(tableName) > -1) {
						schemaCache.aliases[tableName] = tableName;
						fetchTableColumns(tableName);

						if (alias && !reserved[alias]) {
							schemaCache.aliases[alias] = tableName;
							console.log('Schema: Mapped alias ' + alias + ' -> ' + tableName);
						}
					}
				}
			}

			console.log('Schema: Final aliases:', schemaCache.aliases);
		}

		// Parse CTEs from query - extracts CTE names and their columns
		function parseCTEs(query) {
			schemaCache.ctes = {};
			var q = query;
			var qLower = query.toLowerCase();

			// Check if query contains WITH
			var withIdx = qLower.search(/with\\s+/i);
			if (withIdx === -1) return;

			// Find the main SELECT (not inside CTE) - after all CTEs
			var mainSelectIdx = findMainSelect(q);
			if (mainSelectIdx === -1) return;

			var withClause = q.substring(withIdx, mainSelectIdx);

			// Parse each CTE: name AS (SELECT ...)
			var ctePattern = /([a-zA-Z_][a-zA-Z0-9_]*)\\s+as\\s*\\(/gi;
			var match;
			var ctePositions = [];

			while ((match = ctePattern.exec(withClause)) !== null) {
				ctePositions.push({
					name: match[1].toLowerCase(),
					start: match.index + match[0].length
				});
			}

			// For each CTE, find its body and parse columns
			for (var i = 0; i < ctePositions.length; i++) {
				var cte = ctePositions[i];
				var startIdx = cte.start;

				// Find matching closing parenthesis
				var endIdx = findMatchingParen(withClause, startIdx - 1);
				if (endIdx === -1) continue;

				var cteBody = withClause.substring(startIdx, endIdx);
				var columns = parseSelectColumns(cteBody);

				schemaCache.ctes[cte.name] = columns;
				schemaCache.aliases[cte.name] = cte.name;
			}
		}

		// Find the main SELECT statement (after all CTEs)
		function findMainSelect(query) {
			var q = query.toLowerCase();
			var idx = 0;
			var depth = 0;
			var inWith = false;

			// Skip initial whitespace
			while (idx < q.length && /\\s/.test(q[idx])) idx++;

			// Check for WITH keyword
			if (q.substring(idx, idx + 4) === 'with') {
				inWith = true;
				idx += 4;
			} else {
				return q.indexOf('select');
			}

			// Scan through CTEs until we find the main SELECT
			while (idx < q.length) {
				var char = q[idx];

				if (char === '(') {
					depth++;
				} else if (char === ')') {
					depth--;
				} else if (depth === 0) {
					// Check for SELECT at depth 0 (main query)
					if (q.substring(idx, idx + 6) === 'select') {
						return idx;
					}
				}
				idx++;
			}
			return -1;
		}

		// Find matching closing parenthesis
		function findMatchingParen(str, openIdx) {
			var depth = 1;
			var idx = openIdx + 1;

			while (idx < str.length && depth > 0) {
				if (str[idx] === '(') depth++;
				else if (str[idx] === ')') depth--;
				idx++;
			}

			return depth === 0 ? idx - 1 : -1;
		}

		// Parse SELECT columns from a CTE body and extract aliases
		function parseSelectColumns(cteBody) {
			var columns = [];
			var q = cteBody;
			var qLower = cteBody.toLowerCase();

			// Find SELECT keyword
			var selectIdx = qLower.indexOf('select');
			if (selectIdx === -1) return columns;

			// Find FROM keyword (end of column list)
			var fromIdx = findKeywordOutsideParens(qLower, 'from', selectIdx);
			if (fromIdx === -1) return columns;

			var columnPart = q.substring(selectIdx + 6, fromIdx).trim();

			// Split by comma (but not inside parentheses)
			var columnDefs = splitByCommaOutsideParens(columnPart);

			for (var i = 0; i < columnDefs.length; i++) {
				var colDef = columnDefs[i].trim();
				if (!colDef) continue;

				var colName = extractColumnAlias(colDef);
				if (colName) {
					columns.push(colName.toLowerCase());
				}
			}

			return columns;
		}

		// Find a keyword that's not inside parentheses
		function findKeywordOutsideParens(str, keyword, startIdx) {
			var depth = 0;
			var idx = startIdx || 0;
			var keyLen = keyword.length;

			while (idx < str.length - keyLen) {
				var char = str[idx];

				if (char === '(') {
					depth++;
				} else if (char === ')') {
					depth--;
				} else if (depth === 0) {
					// Check for keyword with word boundaries
					if (str.substring(idx, idx + keyLen) === keyword) {
						var before = idx > 0 ? str[idx - 1] : ' ';
						var after = idx + keyLen < str.length ? str[idx + keyLen] : ' ';
						if (/\\s/.test(before) && /[\\s(]/.test(after)) {
							return idx;
						}
					}
				}
				idx++;
			}
			return -1;
		}

		// Split string by comma, ignoring commas inside parentheses
		function splitByCommaOutsideParens(str) {
			var parts = [];
			var current = '';
			var depth = 0;

			for (var i = 0; i < str.length; i++) {
				var char = str[i];

				if (char === '(') {
					depth++;
					current += char;
				} else if (char === ')') {
					depth--;
					current += char;
				} else if (char === ',' && depth === 0) {
					parts.push(current);
					current = '';
				} else {
					current += char;
				}
			}

			if (current.trim()) {
				parts.push(current);
			}

			return parts;
		}

		// Extract column alias from a column definition
		function extractColumnAlias(colDef) {
			var def = colDef.trim();
			var defLower = def.toLowerCase();

			// Check for explicit AS alias
			// Look for " as " or " AS " pattern (not inside function names like "cast")
			var asMatch = def.match(/\\s+as\\s+([a-zA-Z_][a-zA-Z0-9_]*)\\s*$/i);
			if (asMatch) {
				return asMatch[1];
			}

			// Check for implicit alias (last word after space, not inside parens)
			// e.g., "b.class location" -> "location"
			var depth = 0;
			var lastSpaceIdx = -1;

			for (var i = def.length - 1; i >= 0; i--) {
				var char = def[i];
				if (char === ')') depth++;
				else if (char === '(') depth--;
				else if (depth === 0 && /\\s/.test(char)) {
					lastSpaceIdx = i;
					break;
				}
			}

			if (lastSpaceIdx > 0) {
				var potentialAlias = def.substring(lastSpaceIdx + 1).trim();
				// Make sure it's a valid identifier (not a keyword or operator)
				if (/^[a-zA-Z_][a-zA-Z0-9_]*$/.test(potentialAlias)) {
					var keywords = ['and', 'or', 'not', 'in', 'is', 'null', 'between', 'like', 'as', 'on', 'from', 'where'];
					if (keywords.indexOf(potentialAlias.toLowerCase()) === -1) {
						return potentialAlias;
					}
				}
			}

			// If simple column reference like "a.column" or just "column"
			var simpleMatch = def.match(/^([a-zA-Z_][a-zA-Z0-9_]*\\.)?([a-zA-Z_][a-zA-Z0-9_]*)$/);
			if (simpleMatch) {
				return simpleMatch[2];
			}

			return null;
		}

		// Get columns for an identifier (alias, table, or CTE)
		function getColumnsForIdentifier(id) {
			id = id.toLowerCase();

			// Check CTE directly
			if (schemaCache.ctes[id] && schemaCache.ctes[id].length > 0) {
				return schemaCache.ctes[id];
			}

			// Check alias -> get real table/CTE
			var realTable = schemaCache.aliases[id];

			// Check if alias points to a CTE
			if (realTable && schemaCache.ctes[realTable] && schemaCache.ctes[realTable].length > 0) {
				return schemaCache.ctes[realTable];
			}

			// Check if alias points to a table
			if (realTable && schemaCache.tables[realTable]) {
				return schemaCache.tables[realTable];
			}

			// Direct table check
			if (schemaCache.tables[id]) {
				return schemaCache.tables[id];
			}

			// Try to fetch if it's a known table (but not if it's a CTE)
			if (!schemaCache.ctes[id] && !schemaCache.ctes[realTable]) {
				if (schemaCache.tableList.indexOf(id) > -1) {
					fetchTableColumns(id);
				}
				if (realTable && schemaCache.tableList.indexOf(realTable) > -1) {
					fetchTableColumns(realTable);
				}
			}

			return [];
		}

		// Parse query on change
		function parseQueryContext(cm) {
			var query = cm.getValue();
			console.log('Schema: parseQueryContext called');
			parseCTEs(query);
			parseTableAliases(query);
		}

		// Custom SQL hint function
		function sqlHintWithContext(cm, options) {
			parseQueryContext(cm);

			var cursor = cm.getCursor();
			var line = cm.getLine(cursor.line);
			var beforeCursor = line.substring(0, cursor.ch);

			// Find word before cursor (including dot)
			var match = beforeCursor.match(/([a-zA-Z_][a-zA-Z0-9_]*)\\.([a-zA-Z0-9_]*)$/);

			if (match) {
				// alias.column pattern
				var alias = match[1];
				var partial = match[2];

				var columns = getColumnsForIdentifier(alias);

				if (columns.length === 0) {
					// Columns not loaded yet, show loading message
					return {
						list: [{text: '', displayText: 'Loading columns...'}],
						from: cursor,
						to: cursor
					};
				}

				var filtered = [];
				for (var i = 0; i < columns.length; i++) {
					var col = columns[i];
					if (partial === '' || col.indexOf(partial.toLowerCase()) === 0) {
						filtered.push({text: col, displayText: col});
					}
				}

				filtered.sort(function(a, b) {
					return a.text.localeCompare(b.text);
				});

				return {
					list: filtered,
					from: CodeMirror.Pos(cursor.line, cursor.ch - partial.length),
					to: cursor
				};
			}

			// Check if just typed a dot: "alias."
			var dotMatch = beforeCursor.match(/([a-zA-Z_][a-zA-Z0-9_]*)\\.$/);
			if (dotMatch) {
				var alias = dotMatch[1];
				var columns = getColumnsForIdentifier(alias);

				if (columns.length === 0) {
					return {
						list: [{text: '', displayText: 'Loading columns...'}],
						from: cursor,
						to: cursor
					};
				}

				var list = [];
				for (var i = 0; i < columns.length; i++) {
					list.push({text: columns[i], displayText: columns[i]});
				}

				list.sort(function(a, b) {
					return a.text.localeCompare(b.text);
				});

				return {
					list: list,
					from: cursor,
					to: cursor
				};
			}

			// Default: use standard SQL hints
			return CodeMirror.hint.sql(cm, {
				tables: sqlEditor.getOption('hintOptions').tables,
				completeSingle: false
			});
		}
	`

}


function jsFunctionInitCodeMirror() {

	return `

		function initCodeMirror() {
			var editorContainer = document.getElementById('queryEditorContainer');

			sqlEditor = CodeMirror(editorContainer, {
				mode: 'text/x-sql',
				lineNumbers: true,
				matchBrackets: true,
				autoCloseBrackets: true,
				styleActiveLine: true,
				indentWithTabs: false,
				smartIndent: true,
				tabSize: 4,
				indentUnit: 4,
				lineWrapping: false,
				extraKeys: {
					"Ctrl-Space": function(cm) {
						CodeMirror.showHint(cm, sqlHintWithContext, {completeSingle: false});
					},
					"Ctrl-Enter": function(cm) { querySubmit(); },
					"Ctrl-Shift-F": function(cm) { formatQuery(); },
					"Tab": function(cm) {
						if (cm.somethingSelected()) {
							cm.indentSelection("add");
						} else {
							cm.replaceSelection("    ", "end");
						}
					}
				},
				hintOptions: {
					tables: {},
					completeSingle: false
				}
			});

			var initialContent = document.getElementById('query').value;
			if (initialContent) {
				sqlEditor.setValue(initialContent);
			}

			sqlEditor.on('change', function(cm) {
				document.getElementById('query').value = cm.getValue();
				fileInfoRefresh();
				// Parse query context on change to detect table references
				parseQueryContext(cm);
			});

			sqlEditor.on('inputRead', function(cm, change) {
				var typed = change.text[0];
				// Trigger on letters, underscore, or dot
				if (typed && typed.match(/[a-zA-Z_]/)) {
					setTimeout(function() {
						CodeMirror.showHint(cm, sqlHintWithContext, {completeSingle: false});
					}, 50);
				}
				// Special handling for dot - trigger column autocomplete
				if (typed === '.') {
					setTimeout(function() {
						CodeMirror.showHint(cm, sqlHintWithContext, {completeSingle: false});
					}, 100);
				}
			});

			sqlEditor.focus();

			// Start prefetching table names in background
			setTimeout(prefetchTableNames, 100);

			// Initialize editor resize functionality
			initEditorResize();
		}

		function initEditorResize() {
			var resizeHandle = document.getElementById('editorResizeHandle');
			var editorContainer = document.getElementById('queryEditorContainer');

			if (!resizeHandle || !editorContainer) return;

			var isResizing = false;
			var startY = 0;
			var startHeight = 0;
			var minHeight = 150;
			var maxHeight = 800;

			resizeHandle.addEventListener('mousedown', function(e) {
				isResizing = true;
				startY = e.clientY;
				startHeight = editorContainer.offsetHeight;

				document.body.style.cursor = 'ns-resize';
				document.body.style.userSelect = 'none';

				e.preventDefault();
			});

			document.addEventListener('mousemove', function(e) {
				if (!isResizing) return;

				var deltaY = e.clientY - startY;
				var newHeight = Math.min(maxHeight, Math.max(minHeight, startHeight + deltaY));

				editorContainer.style.height = newHeight + 'px';

				// Refresh CodeMirror to adjust to new size
				if (sqlEditor) {
					sqlEditor.refresh();
				}
			});

			document.addEventListener('mouseup', function() {
				if (isResizing) {
					isResizing = false;
					document.body.style.cursor = '';
					document.body.style.userSelect = '';
				}
			});

			// Touch support for mobile devices
			resizeHandle.addEventListener('touchstart', function(e) {
				isResizing = true;
				startY = e.touches[0].clientY;
				startHeight = editorContainer.offsetHeight;
				e.preventDefault();
			}, { passive: false });

			document.addEventListener('touchmove', function(e) {
				if (!isResizing) return;

				var deltaY = e.touches[0].clientY - startY;
				var newHeight = Math.min(maxHeight, Math.max(minHeight, startHeight + deltaY));

				editorContainer.style.height = newHeight + 'px';

				if (sqlEditor) {
					sqlEditor.refresh();
				}
			}, { passive: true });

			document.addEventListener('touchend', function() {
				if (isResizing) {
					isResizing = false;
				}
			});
		}

	`

}


function jsFunctionSQLFormatter() {

	return `
	
		var SQLFormatter = {
			
			keywords: [
				'SELECT', 'FROM', 'WHERE', 'AND', 'OR', 'NOT', 'IN', 'EXISTS',
				'JOIN', 'LEFT', 'RIGHT', 'INNER', 'OUTER', 'CROSS', 'FULL',
				'ON', 'AS', 'ORDER', 'BY', 'GROUP', 'HAVING', 'LIMIT', 'OFFSET',
				'UNION', 'ALL', 'DISTINCT', 'TOP', 'INTO', 'INSERT', 'UPDATE',
				'DELETE', 'SET', 'VALUES', 'CREATE', 'ALTER', 'DROP', 'TABLE',
				'INDEX', 'VIEW', 'CASE', 'WHEN', 'THEN', 'ELSE', 'END', 'IS',
				'NULL', 'LIKE', 'BETWEEN', 'ASC', 'DESC', 'NULLS', 'FIRST', 'LAST',
				'WITH', 'RECURSIVE', 'OVER', 'PARTITION', 'ROWS', 'RANGE',
				'UNBOUNDED', 'PRECEDING', 'FOLLOWING', 'CURRENT', 'ROW',
				'COUNT', 'SUM', 'AVG', 'MIN', 'MAX', 'COALESCE', 'NVL', 'NVL2',
				'CAST', 'CONVERT', 'SUBSTR', 'SUBSTRING', 'TRIM', 'UPPER', 'LOWER',
				'ROWNUM', 'BUILTIN', 'DF', 'CONNECT_BY_ROOT', 'PRIOR', 'LEVEL',
				'START', 'CONNECT'
			],
			
			format: function(sql) {
				if (!sql || sql.trim() === '') return '';
				
				// Preserve string literals
				var strings = [];
				sql = sql.replace(/'([^']*)'|"([^"]*)"/g, function(match) {
					strings.push(match);
					return '___STRING' + (strings.length - 1) + '___';
				});
				
				// Preserve comments
				var comments = [];
				sql = sql.replace(/--[^\\n]*/g, function(match) {
					comments.push(match);
					return '___COMMENT' + (comments.length - 1) + '___';
				});
				
				// Normalize whitespace
				sql = sql.replace(/\\s+/g, ' ').trim();
				
				// Uppercase keywords
				this.keywords.forEach(function(keyword) {
					var regex = new RegExp('\\\\b' + keyword + '\\\\b', 'gi');
					sql = sql.replace(regex, keyword);
				});
				
				// Handle multi-word keywords
				sql = sql.replace(/ORDER\\s+BY/gi, 'ORDER BY');
				sql = sql.replace(/GROUP\\s+BY/gi, 'GROUP BY');
				sql = sql.replace(/LEFT\\s+JOIN/gi, 'LEFT JOIN');
				sql = sql.replace(/RIGHT\\s+JOIN/gi, 'RIGHT JOIN');
				sql = sql.replace(/INNER\\s+JOIN/gi, 'INNER JOIN');
				sql = sql.replace(/OUTER\\s+JOIN/gi, 'OUTER JOIN');
				sql = sql.replace(/CROSS\\s+JOIN/gi, 'CROSS JOIN');
				sql = sql.replace(/FULL\\s+JOIN/gi, 'FULL JOIN');
				sql = sql.replace(/IS\\s+NULL/gi, 'IS NULL');
				sql = sql.replace(/IS\\s+NOT\\s+NULL/gi, 'IS NOT NULL');
				sql = sql.replace(/NOT\\s+IN/gi, 'NOT IN');
				sql = sql.replace(/NOT\\s+EXISTS/gi, 'NOT EXISTS');
				sql = sql.replace(/NOT\\s+LIKE/gi, 'NOT LIKE');
				
				// Add newlines (using 4 spaces for indentation)
				sql = sql.replace(/\\s+(SELECT)\\s+/gi, '\\nSELECT\\n    ');
				sql = sql.replace(/\\s+(FROM)\\s+/gi, '\\nFROM\\n    ');
				sql = sql.replace(/\\s+(WHERE)\\s+/gi, '\\nWHERE\\n    ');
				sql = sql.replace(/\\s+(AND)\\s+/gi, '\\n    AND ');
				sql = sql.replace(/\\s+(OR)\\s+/gi, '\\n    OR ');
				sql = sql.replace(/\\s+(ORDER BY)\\s+/gi, '\\nORDER BY\\n    ');
				sql = sql.replace(/\\s+(GROUP BY)\\s+/gi, '\\nGROUP BY\\n    ');
				sql = sql.replace(/\\s+(HAVING)\\s+/gi, '\\nHAVING\\n    ');
				sql = sql.replace(/\\s+(LIMIT)\\s+/gi, '\\nLIMIT ');
				sql = sql.replace(/\\s+(OFFSET)\\s+/gi, '\\nOFFSET ');
				sql = sql.replace(/\\s+(UNION ALL|UNION)\\s+/gi, '\\n\\n$1\\n\\n');
				
				// JOINs
				sql = sql.replace(/\\s+(LEFT JOIN|RIGHT JOIN|INNER JOIN|OUTER JOIN|CROSS JOIN|FULL JOIN|JOIN)\\s+/gi, '\\n$1\\n    ');
				sql = sql.replace(/\\s+(ON)\\s+/gi, '\\n        ON ');
				
				// CASE statements
				sql = sql.replace(/\\s+(CASE)\\s+/gi, '\\n    CASE');
				sql = sql.replace(/\\s+(WHEN)\\s+/gi, '\\n        WHEN ');
				sql = sql.replace(/\\s+(THEN)\\s+/gi, ' THEN ');
				sql = sql.replace(/\\s+(ELSE)\\s+/gi, '\\n        ELSE ');
				sql = sql.replace(/\\s+(END)([\\s,])/gi, '\\n    END$2');
				
				// WITH clause
				sql = sql.replace(/^\\s*(WITH)\\s+/gi, 'WITH\\n    ');
				
				// Format commas
				sql = sql.replace(/,\\s*/g, ',\\n    ');
				
				// Clean up
				sql = sql.replace(/\\n\\s*\\n/g, '\\n');
				sql = sql.replace(/^\\s+/gm, function(match) {
					var spaces = match.match(/ /g);
					return spaces ? ' '.repeat(Math.min(spaces.length, 12)) : '';
				});
				
				// Restore strings
				strings.forEach(function(str, i) {
					sql = sql.replace('___STRING' + i + '___', str);
				});
				
				// Restore comments
				comments.forEach(function(comment, i) {
					sql = sql.replace('___COMMENT' + i + '___', comment);
				});
				
				return sql.trim();
			}
		};
	
	`

}


function jsFunctionFormatQuery() {

	return `
	
		function formatQuery() {
			if (!sqlEditor) return;
			
			var currentSQL = sqlEditor.getValue();
			if (currentSQL.trim() === '') {
				alert('Please enter a query to format.');
				return;
			}
			
			var formattedSQL = SQLFormatter.format(currentSQL);
			var cursor = sqlEditor.getCursor();
			
			sqlEditor.setValue(formattedSQL);
			
			var lineCount = sqlEditor.lineCount();
			if (cursor.line < lineCount) {
				sqlEditor.setCursor(cursor.line, 0);
			}
			
			sqlEditor.focus();
		}
	
	`

}


function jsFunctionToggleTheme() {

	return `
	
		function toggleTheme() {
			isDarkTheme = !isDarkTheme;
			var container = document.getElementById('queryEditorContainer');
			
			if (isDarkTheme) {
				container.classList.add('cm-dark');
			} else {
				container.classList.remove('cm-dark');
			}
			
			try {
				localStorage.setItem('suiteql_dark_theme', isDarkTheme);
			} catch(e) {}
		}
		
		$(document).ready(function() {
			try {
				var savedDark = localStorage.getItem('suiteql_dark_theme');
				if (savedDark === 'true') {
					isDarkTheme = true;
					var container = document.getElementById('queryEditorContainer');
					if (container) container.classList.add('cm-dark');
				}
			} catch(e) {}
		});
	
	`

}


function jsFunctionDataTablesExternals() {

	if ( datatablesEnabled === true ) {

		return `
			<link rel="stylesheet" type="text/css" href="${getLibUrl('jquery.dataTables.min.css')}">
			<script type="text/javascript" charset="utf8" src="${getLibUrl('jquery.dataTables.min.js')}"></script>
		`

	} else {

		return ``

	}

}


function jsFunctionDefaultQuerySet() {

	return `
		function defaultQuerySet() {	
			var defaultSQL = 'SELECT\\n    ID,\\n    LastName,\\n    FirstName,\\n    Phone,\\n    Email\\nFROM\\n    Employee\\nWHERE\\n    Email LIKE \\'%@test.com\\'\\nORDER BY\\n    LastName,\\n    FirstName';
			
			if (sqlEditor) {
				sqlEditor.setValue(defaultSQL);
				sqlEditor.focus();
			} else {
				document.getElementById('query').value = defaultSQL;
			}
			return false;
		}	
	`
	
}


function jsFunctionDocumentGenerate() {

	return `
	
		function documentGenerate() {
		
			var queryValue = sqlEditor ? sqlEditor.getValue() : document.getElementById('query').value;
	
			if (queryValue == '') { 
				alert('Please enter a query.');
				return; 
			}
		
			var rowBegin, rowEnd;
			
			if (document.getElementById('returnAll').checked) {
				rowBegin = 1;
				rowEnd = 999999;
			} else {
				rowBegin = parseInt(document.getElementById('rowBegin').value);
				if (!Number.isInteger(rowBegin)) {
					alert('Enter an integer for the beginning row.');
					return;
				}
				rowEnd = parseInt(document.getElementById('rowEnd').value);
				if (!Number.isInteger(rowEnd)) {
					alert('Enter an integer for the ending row.');
					return;
				}	
			}	
			
			if (document.getElementById('template').value == '') { 
				alert('Please enter a template.');
				return; 
			}	
			
			var viewsEnabled = document.getElementById('enableViews') ? document.getElementById('enableViews').checked : false;
								
			var requestPayload = { 
				'function': 'documentSubmit', 
				'query': queryValue,
				'rowBegin': rowBegin,
				'rowEnd': rowEnd,
				'viewsEnabled': viewsEnabled,
				'returnTotals': document.getElementById('returnTotals').checked,				
				'template': document.getElementById('template').value,
				'docType': radioFieldValueGet('resultsFormat')
			}
			
			var xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', true);
			xhr.setRequestHeader('Accept', 'application/json');		
			xhr.send(JSON.stringify(requestPayload));
		
			xhr.onload = function() {
				if (xhr.status === 200) {	
					try {			
						queryResponsePayload = JSON.parse(xhr.response);						
					} catch(e) {	
						alert('Unable to parse the response.');
						return;					
					}
			
					if (queryResponsePayload['error'] == undefined) {				
						window.open('${scriptURL}&function=documentGenerate');
					} else {					
						alert('Error: ' + queryResponsePayload.error.message);									
					}
				} else {				
					alert('Error: ' + xhr.status);									
				}
			}
		}
	`

}


function jsFunctionEnablePaginationToggle() {

	return `
		function enablePaginationToggle() {
			var opts = document.getElementById('paginationOptions');
			if (document.getElementById('enablePagination').checked) {
				opts.style.display = 'block';
				returnAllToggle();
			} else {
				opts.style.display = 'none';
			}
		}	
	`

}	


function jsFunctionFileInfoRefresh() {

	return `
		function fileInfoRefresh() {
			var content = '';
			var queryValue = sqlEditor ? sqlEditor.getValue() : document.getElementById('query').value;
			
			if (activeSQLFile.source == undefined) {	
				if (queryValue != '') {
					content = '<span class="text-danger">unsaved</span>';
				}
			} else {
				var status = queryValue != activeSQLFile.sql ? 'modified' : 'saved';
				content = '<span title="Source: ' + activeSQLFile.source + '">' + activeSQLFile.fileName + '</span>';
				if (status === 'modified') {
					content = '<span class="text-danger">' + content + ' (modified)</span>';
				}
			}
			
			document.getElementById('fileInfo').innerHTML = content;
		}	
	`
	
}


function jsFunctionHideRowNumbersToggle() {

	return `
		function hideRowNumbersToggle() {
			responseGenerate();
		}	
	`

}


function jsFunctionLocalLibraryButtons() {

	if ( queryFolderID !== null ) {
		return `<button type="button" class="btn btn-sm btn-light" data-toggle="modal" data-target="#localLoadModal">Local</button>									
		<button type="button" class="btn btn-sm btn-light" data-toggle="modal" data-target="#saveModal">Save</button>`
	} else {
		return ``
	}

}


function jsFunctionLocalLibraryFilesGet() {

	return `
	
		function localLibraryFilesGet() {	
			document.getElementById('localSQLFilesList').innerHTML = '<div class="loading-overlay"><div class="spinner"></div><p>Loading files...</p></div>';											
									
			var requestPayload = { 'function': 'localLibraryFilesGet' }
			var xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', true);
			xhr.send(JSON.stringify(requestPayload));
			
			xhr.onload = function() {
				var content = '';
			
				if (xhr.status === 200) {	
					payload = JSON.parse(xhr.response);
				
					if (payload.error == undefined) {	
						content = '<div class="table-responsive">';
						content += '<table id="localFilesTable" class="table table-sm table-bordered table-hover">';
						content += '<thead><tr><th>Name</th><th>Description</th><th></th></tr></thead>';
						content += '<tbody>';
						for (var r = 0; r < payload.records.length; r++) {	
							var desc = payload.records[r].description || '';
							content += '<tr>';
							content += '<td>' + payload.records[r].name + '</td>';
							content += '<td>' + desc + '</td>';
							content += '<td style="text-align: center;"><button type="button" class="btn btn-sm btn-primary" onclick="localSQLFileLoad(' + payload.records[r].id + ');">Load</button></td>';
							content += '</tr>';		
						}	
						content += '</tbody></table></div>';
						
						document.getElementById('localSQLFilesList').innerHTML = content;
						if (${datatablesEnabled}) { $('#localFilesTable').DataTable(); }							
					} else {
						content = payload.error == 'No SQL Files' 
							? '<p style="color: #ef4444;">No query files found in the local folder.</p>'
							: '<p style="color: #ef4444;">Error: ' + payload.error + '</p>';
						document.getElementById('localSQLFilesList').innerHTML = content;
					}																							
				} else {
					document.getElementById('localSQLFilesList').innerHTML = '<p style="color: #ef4444;">Error: ' + xhr.status + '</p>';		
				}
			}															
		}	
	`

}


function jsFunctionLocalSQLFileLoad() {

	return `
	
		function localSQLFileLoad(fileID) {
			var requestPayload = { 'function': 'sqlFileLoad', 'fileID': fileID }
			var xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', true);
			xhr.send(JSON.stringify(requestPayload));
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					fileLoadResponsePayload = JSON.parse(xhr.response);	
					
					if (fileLoadResponsePayload.error == undefined) {									
						if (sqlEditor) {
							sqlEditor.setValue(fileLoadResponsePayload.sql);
						} else {
							document.getElementById('query').value = fileLoadResponsePayload.sql;
						}
						
						$('#localLoadModal').modal('toggle');
						document.getElementById('resultsDiv').style.display = "none";									
						
						activeSQLFile.source = 'Local SQL Library';
						activeSQLFile.fileName = fileLoadResponsePayload.file.name;
						activeSQLFile.description = fileLoadResponsePayload.file.description;
						activeSQLFile.fileID = fileLoadResponsePayload.file.id;
						activeSQLFile.sql = fileLoadResponsePayload.sql;
						fileInfoRefresh();															
					} else {
						alert('Error: ' + fileLoadResponsePayload.error);
					}																							
				} else {
					alert('Error: ' + xhr.status);								
				}
			}							
		}	
	`

}


function jsFunctionLocalSQLFileSave() {

	return `
	
		function localSQLFileSave() {
			var filename = document.getElementById('saveQueryFormFileName').value;
			if (filename == '') {
				alert('Please enter a name for the file.');
				return;
			}
			
			// Check if exists
			var requestPayload = { 'function': 'sqlFileExists', 'filename': filename }
			var xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', false);
			xhr.send(JSON.stringify(requestPayload));		
			
			if (xhr.status === 200) {	
				var existsPayload = JSON.parse(xhr.response);	
				if (existsPayload.exists == true) {									
   					if (!confirm('A file named "' + filename + '" already exists. Replace it?')) {
   						return;
   					}
				}																							
			} else {
				alert('Error: ' + xhr.status);	
				return;							
			}					
			
			var queryValue = sqlEditor ? sqlEditor.getValue() : document.getElementById('query').value;				
			
			requestPayload = { 
				'function': 'sqlFileSave',
				'filename': filename,
				'contents': queryValue,
				'description': document.getElementById('saveQueryFormDescription').value
			}

			xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', true);
			xhr.send(JSON.stringify(requestPayload));
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					var savePayload = JSON.parse(xhr.response);	
					
					if (savePayload.error == undefined) {		
						activeSQLFile.source = 'Local SQL Library';
						activeSQLFile.fileName = filename;
						activeSQLFile.description = document.getElementById('saveQueryFormDescription').value;
						activeSQLFile.fileID = savePayload.fileID;
						activeSQLFile.sql = queryValue;
						fileInfoRefresh();													
						alert('File saved successfully.');										
					} else {
						alert('Error: ' + savePayload.error);
					}																							
				} else {
					alert('Error: ' + xhr.status);								
				}
			}		
			
			$('#saveModal').modal('toggle');
		}				
	`

}


function jsFunctionQueryFormRowToggle() {
	return `
		function queryFormRowToggle() {	
			var row = document.getElementById('queryFormRow');
			var btn = document.getElementById('btnQueryFormRowToggle');
			
			if (row.style.display !== 'none') {
				row.style.display = 'none';
				btn.textContent = 'Show Editor';
			} else {
				row.style.display = 'flex';
				btn.textContent = 'Hide Editor';
				if (sqlEditor) sqlEditor.refresh();
			}
		}
	`
}


function jsFunctionQueryProgress() {

	return `
		function showQueryProgress() {
			// Remove any existing popup
			hideQueryProgress();

			var popup = document.createElement('div');
			popup.id = 'queryProgressPopup';
			popup.className = 'query-progress-popup';
			popup.innerHTML = '<div class="spinner-small"></div>' +
				'<div style="flex: 1;">' +
				'<div class="progress-text">Running query...</div>' +
				'<div class="progress-bar-container"><div class="progress-bar"></div></div>' +
				'</div>';
			document.body.appendChild(popup);
		}

		function hideQueryProgress() {
			var popup = document.getElementById('queryProgressPopup');
			if (popup) {
				popup.classList.add('hiding');
				setTimeout(function() {
					if (popup.parentNode) {
						popup.parentNode.removeChild(popup);
					}
				}, 300);
			}
		}
	`
}


function jsFunctionQuerySubmit() {

	return `

		function querySubmit() {
			var queryValue = sqlEditor ? sqlEditor.getValue() : document.getElementById('query').value;
			
			if (queryValue == '') { 
				alert('Please enter a query.');
				return; 
			}
			
			var theQuery;
			if (sqlEditor) {
				var selectedText = sqlEditor.getSelection();
				theQuery = selectedText !== '' ? selectedText : queryValue;
			} else {
				var textArea = document.getElementById('query');
				if (textArea.selectionStart !== undefined) {
					theQuery = textArea.value.substring(textArea.selectionStart, textArea.selectionEnd);
				}
				if (!theQuery) theQuery = queryValue;
			}

			var rowBegin, rowEnd;
			if (document.getElementById('returnAll') && document.getElementById('returnAll').checked) {
				rowBegin = 1;
				rowEnd = 999999;
			} else {
				rowBegin = parseInt(document.getElementById('rowBegin').value) || 1;
				rowEnd = parseInt(document.getElementById('rowEnd').value) || 25;
			}
			
			var viewsEnabled = document.getElementById('enableViews') ? document.getElementById('enableViews').checked : false;
			var paginationEnabled = document.getElementById('enablePagination').checked;

			// Show non-blocking popup progress bar
			showQueryProgress();

			var requestPayload = { 
				'function': 'queryExecute', 
				'query': theQuery,
				'rowBegin': rowBegin,
				'rowEnd': rowEnd,
				'paginationEnabled': paginationEnabled,
				'viewsEnabled': viewsEnabled,
				'returnTotals': document.getElementById('returnTotals') ? document.getElementById('returnTotals').checked : false
			}

			var xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', true);
			xhr.setRequestHeader('Accept', 'application/json');		
			xhr.send(JSON.stringify(requestPayload));
		
			xhr.onload = function() {
				hideQueryProgress();
				document.getElementById('resultsDiv').style.display = "block";

				if (xhr.status === 200) {
					try {
						queryResponsePayload = JSON.parse(xhr.response);
					} catch(e) {
						document.getElementById('resultsDiv').innerHTML = '<p style="color: #ef4444;">Unable to parse the response.</p>';
						return;
					}

					if (queryResponsePayload['error'] == undefined) {
						responseGenerate();
					} else {
						document.getElementById('resultsDiv').innerHTML = '<div style="background: #fef2f2; border: 1px solid #fecaca; border-radius: 6px; padding: 16px;"><p style="color: #dc2626; margin: 0; font-weight: 500;">Error</p><pre style="margin: 8px 0 0; color: #7f1d1d; font-size: 12px;">' + queryResponsePayload.error.message + '</pre></div>';
					}
				} else {
					document.getElementById('resultsDiv').innerHTML = '<p style="color: #ef4444;">Request failed: ' + xhr.status + '</p>';
				}
			};

			xhr.onerror = function() {
				hideQueryProgress();
				document.getElementById('resultsDiv').style.display = "block";
				document.getElementById('resultsDiv').innerHTML = '<p style="color: #ef4444;">Network error occurred.</p>';
			};
		}
	`


}


function jsFunctionQueryTextAreaResize() {

	return `
		function queryTextAreaResize() {					
			if (sqlEditor) sqlEditor.refresh();
		}
	`	

}


function jsFunctionRadioFieldValueGet() {

	return `
		function radioFieldValueGet(fieldName) {					
			var radios = document.getElementsByName(fieldName);
			for (var i = 0; i < radios.length; i++) {						
			  if (radios[i].checked) return radios[i].value;
			}						
			return '';						
		}		
	`

}


function jsFunctionRemoteLibraryButton() {

	if ( remoteLibraryEnabled === true ) {
		return `<button type="button" class="btn btn-sm btn-light" data-toggle="modal" data-target="#remoteLoadModal">Remote</button>`
	} else {
		return ``
	}

}


function jsFunctionRemoteLibraryIndexGet() {

	return `
	
		function remoteLibraryIndexGet() {											
			document.getElementById('remoteSQLFilesList').innerHTML = '<div class="loading-overlay"><div class="spinner"></div><p>Loading library...</p></div>';											

			var xhr = new XMLHttpRequest();
			xhr.open('GET', 'https://suiteql.s3.us-east-1.amazonaws.com/queries/index.json?nonce=' + new Date().getTime(), true);
			xhr.send();
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					var payload = JSON.parse(xhr.response);
				
					var content = '<div class="table-responsive">';
					content += '<table class="table table-sm table-bordered table-hover" id="remoteFilesTable">';
					content += '<thead><tr><th>Name</th><th>Description</th><th></th></tr></thead>';
					content += '<tbody>';
					for (var r = 0; r < payload.length; r++) {	
						content += '<tr>';
						content += '<td width="35%">' + payload[r].name + '</td>';
						content += '<td>' + payload[r].description + '</td>';
						content += '<td style="text-align: center;"><button type="button" class="btn btn-sm btn-primary" onclick="remoteSQLFileLoad(\\'' + payload[r].fileName + '\\');">Load</button></td>';
						content += '</tr>';													
					}	
					content += '</tbody></table></div>';
					
					document.getElementById('remoteSQLFilesList').innerHTML = content;
					if (${datatablesEnabled}) { $('#remoteFilesTable').DataTable(); }																											
				} else {									
					document.getElementById('remoteSQLFilesList').innerHTML = '<p style="color: #ef4444;">Error: ' + xhr.status + '</p>';
				}								
			}															
		}
	`
	
}


function jsFunctionRemoteSQLFileLoad() {

	return `

		function remoteSQLFileLoad(filename) {											
			var xhr = new XMLHttpRequest();
			xhr.open('GET', 'https://suiteql.s3.us-east-1.amazonaws.com/queries/' + filename + '?nonce=' + new Date().getTime(), true);
			xhr.send();
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					if (sqlEditor) {
						sqlEditor.setValue(xhr.response);
					} else {
						document.getElementById('query').value = xhr.response;
					}
					
					$('#remoteLoadModal').modal('toggle');
					document.getElementById('resultsDiv').style.display = "none";			
					
					activeSQLFile.source = 'Remote SQL Library';
					activeSQLFile.fileName = filename;
					activeSQLFile.sql = xhr.response;
					fileInfoRefresh();																																				
				} else {
					alert('Error: ' + xhr.status);
				}
			}															
		}
	`

}


function jsFunctionResponseDataCopy() {

	return `
		function responseDataCopy() {
			var copyText = document.getElementById("responseData");
			copyText.select(); 
			document.execCommand("copy");
			return false;
		}		
	`

}


function jsFunctionResponseGenerate() {

	return `
		function responseGenerate() {
			$('#templateHeaderRow').hide();
			$('#templateFormRow').hide();			
		
			var format = radioFieldValueGet('resultsFormat');
			
			if (format === 'csv') {
				responseGenerateCSV();
			} else if (format === 'json') {
				responseGenerateJSON();
			} else if (format === 'pdf' || format === 'html') {
				$('#templateHeaderRow').show();
				$('#templateFormRow').show();
				responseGenerateTable();
			} else if (format === 'virtual') {
				responseGenerateVirtualScroll();
			} else {
				responseGenerateTable();
			}
		}	
	`

}


function jsFunctionResponseGenerateCSV() {

	return `
	
		function responseGenerateCSV() {		
			document.getElementById('nullFormatDiv').style.display = "none";
			
			if (!queryResponsePayload || !queryResponsePayload.records || queryResponsePayload.records.length === 0) {
				document.getElementById('resultsDiv').innerHTML = '<p style="color: #f59e0b; font-weight: 500;">No records found.</p>';
				return;
			}
												
			var columnNames = Object.keys(queryResponsePayload.records[0]);	
			var csv = '"' + columnNames.join('","') + '"\\r\\n';

			for (var r = 0; r < queryResponsePayload.records.length; r++) {
				var record = queryResponsePayload.records[r];
				var values = [];
				for (var c = 0; c < columnNames.length; c++) {
					var value = record[columnNames[c]];
					value = value != null ? String(value).replace(/"/g, '""') : '';
					values.push('"' + value + '"');
				}
				csv += values.join(',') + '\\r\\n';		
			}	
			
			var content = buildStatsBar();
			content += '<p style="margin-bottom: 8px;"><a href="#" onclick="responseDataCopy(); return false;">Copy to clipboard</a></p>';
			content += '<textarea class="form-control" id="responseData" rows="20" style="font-family: var(--font-code); font-size: 12px;">' + csv + '</textarea>';
									
			document.getElementById('resultsDiv').innerHTML = content;							
		}	
	`

}


function jsFunctionResponseGenerateJSON() {

	return `
	
		function responseGenerateJSON() {	
			document.getElementById('nullFormatDiv').style.display = "none";	
			
			if (!queryResponsePayload || !queryResponsePayload.records || queryResponsePayload.records.length === 0) {
				document.getElementById('resultsDiv').innerHTML = '<p style="color: #f59e0b; font-weight: 500;">No records found.</p>';
				return;
			}
		
			var content = buildStatsBar();
			content += '<p style="margin-bottom: 8px;"><a href="#" onclick="responseDataCopy(); return false;">Copy to clipboard</a></p>';
			content += '<textarea class="form-control" id="responseData" rows="20" style="font-family: var(--font-code); font-size: 12px;">' + JSON.stringify(queryResponsePayload.records, null, 2) + '</textarea>';
			
			document.getElementById('resultsDiv').innerHTML = content;							
		}	
		
		function buildStatsBar() {
			var html = '<div class="stats-bar">';
			html += '<div class="stat-item"><span>Rows:</span> <span class="stat-value">' + queryResponsePayload.records.length + '</span></div>';
			if (document.getElementById('returnTotals') && document.getElementById('returnTotals').checked && queryResponsePayload.totalRecordCount) {
				html += '<div class="stat-item"><span>Total:</span> <span class="stat-value">' + queryResponsePayload.totalRecordCount + '</span></div>';
			}
			html += '<div class="stat-item"><span>Time:</span> <span class="stat-value">' + queryResponsePayload.elapsedTime + 'ms</span></div>';
			html += '</div>';
			return html;
		}
	`

}


function jsFunctionResponseGenerateTable() {

	return `
	
		function responseGenerateTable() {
			document.getElementById('nullFormatDiv').style.display = "block";
			
			if (!queryResponsePayload || !queryResponsePayload.records || queryResponsePayload.records.length === 0) {
				document.getElementById('resultsDiv').innerHTML = '<p style="color: #f59e0b; font-weight: 500;">No records found.</p>';
				return;
			}
					
			var columnNames = Object.keys(queryResponsePayload.records[0]);
			var firstColumnIsRowNumber = document.getElementById('enablePagination').checked;
			var rowNumbersHidden = document.getElementById('hideRowNumbers') && document.getElementById('hideRowNumbers').checked;

			var thead = '<thead><tr>';
			for (var i = 0; i < columnNames.length; i++) {
				if (i == 0 && firstColumnIsRowNumber && rowNumbersHidden) continue;
				if (i == 0 && firstColumnIsRowNumber) {
					thead += '<th style="text-align: center; width: 50px;">#</th>';
				} else {
					thead += '<th>' + columnNames[i] + '</th>';
				}
			}
			thead += '</tr></thead>';

			var nullFormat = radioFieldValueGet('nullFormat');
			var tbody = '<tbody>';
			for (var r = 0; r < queryResponsePayload.records.length; r++) {		
				tbody += '<tr>';
				for (var i = 0; i < columnNames.length; i++) {
					if (i == 0 && firstColumnIsRowNumber && rowNumbersHidden) continue;
					
					var value = queryResponsePayload.records[r][columnNames[i]];
					if (value === null) {
						if (nullFormat == 'dimmed') {
							value = '<span style="color: #9ca3af;">null</span>';
						} else if (nullFormat == 'blank') {
							value = '';
						} else {
							value = 'null';
						}
					} else {
						value = String(value).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
					}
					
					if (i == 0 && firstColumnIsRowNumber) {
						tbody += '<td style="text-align: center;">' + value + '</td>';
					} else {
						tbody += '<td>' + value + '</td>';					
					}
				}				
				tbody += '</tr>';		
			}	
			tbody += '</tbody>';
		
			var content = '';

			// Toolbar with search and stats
			if (radioFieldValueGet('resultsFormat') !== 'datatable') {
				content += '<div class="table-toolbar">';
				content += '<div class="table-search-container">';
				content += '<input type="text" id="tableSearchInput" class="table-search-input" placeholder="Search..." oninput="filterTableRows()">';
				content += '<select id="columnFilterSelect" class="column-filter-select" onchange="filterTableRows()"><option value="">All Columns</option></select>';
				content += '<span id="searchStats" class="search-stats"></span>';
				content += '</div>';
				content += buildStatsBar();
				content += '</div>';
			} else {
				content += buildStatsBar();
			}

			content += '<div class="table-responsive">';
			content += '<table class="results-table" id="resultsTable">';
			content += thead + tbody;
			content += '</table></div>';

			document.getElementById('resultsDiv').innerHTML = content;

			if (radioFieldValueGet('resultsFormat') == 'datatable') {
				var useDeferredRendering = document.getElementById('useDeferredRendering').checked;
				$('#resultsTable').DataTable({
					deferRender: useDeferredRendering,
					pageLength: parseInt(document.getElementById('virtualPageSize').value) || 100,
					scrollX: true,
					scrollY: '400px',
					scrollCollapse: true
				});
			} else {
				// Initialize sorting and column resize for non-DataTable formats
				initTableFeatures('resultsTable');
				// Populate column filter dropdown
				populateColumnFilter('resultsTable');
			}
		}
	`

}


function jsFunctionTableSearch() {

	return `
		function filterTableRows() {
			var table = document.getElementById('resultsTable');
			if (!table) return;

			var searchInput = document.getElementById('tableSearchInput');
			var columnSelect = document.getElementById('columnFilterSelect');
			var searchTerm = searchInput ? searchInput.value : '';
			var selectedColumn = columnSelect ? columnSelect.value : '';

			var tbody = table.getElementsByTagName('tbody')[0];
			if (!tbody) return;

			var rows = tbody.getElementsByTagName('tr');
			var searchLower = searchTerm.toLowerCase().trim();
			var visibleCount = 0;
			var totalCount = rows.length;

			for (var i = 0; i < rows.length; i++) {
				var row = rows[i];
				var cells = row.getElementsByTagName('td');
				var matchText = '';

				if (selectedColumn !== '' && !isNaN(parseInt(selectedColumn))) {
					// Filter by specific column
					var colIndex = parseInt(selectedColumn);
					if (cells[colIndex]) {
						matchText = cells[colIndex].textContent.toLowerCase();
					}
				} else {
					// Filter all columns
					for (var j = 0; j < cells.length; j++) {
						matchText += cells[j].textContent.toLowerCase() + ' ';
					}
				}

				if (searchLower === '' || matchText.indexOf(searchLower) > -1) {
					row.classList.remove('hidden-row');
					visibleCount++;
				} else {
					row.classList.add('hidden-row');
				}
			}

			// Update search stats
			var statsEl = document.getElementById('searchStats');
			if (statsEl) {
				if (searchLower !== '') {
					statsEl.textContent = '(' + visibleCount + '/' + totalCount + ')';
				} else {
					statsEl.textContent = '';
				}
			}
		}

		function populateColumnFilter(tableId) {
			var table = document.getElementById(tableId);
			var select = document.getElementById('columnFilterSelect');
			if (!table || !select) return;

			var headers = table.querySelectorAll('thead th');
			select.innerHTML = '<option value="">All Columns</option>';

			headers.forEach(function(th, index) {
				var columnName = th.textContent.trim().replace(/[\\u2195\\u25B2\\u25BC]/g, '').trim();
				if (columnName && columnName !== '#') {
					var option = document.createElement('option');
					option.value = index;
					option.textContent = columnName;
					select.appendChild(option);
				}
			});
		}
	`

}


function jsFunctionTableSortAndResize() {

	return `
		// Initialize table sorting and column resize
		function initTableFeatures(tableId) {
			var table = document.getElementById(tableId);
			if (!table) return;

			var headers = table.querySelectorAll('thead th');
			headers.forEach(function(th, index) {
				// Add sortable class and resizer
				th.classList.add('sortable');
				th.setAttribute('data-col-index', index);

				// Add resizer element
				var resizer = document.createElement('div');
				resizer.className = 'resizer';
				th.appendChild(resizer);

				// Column resize functionality
				initColumnResize(th, resizer, table);

				// Sort functionality
				th.addEventListener('click', function(e) {
					if (e.target.classList.contains('resizer')) return;
					sortTable(table, index, th);
				});
			});
		}

		function initColumnResize(th, resizer, table) {
			var startX, startWidth, isResizing = false;

			resizer.addEventListener('mousedown', function(e) {
				e.preventDefault();
				e.stopPropagation();
				isResizing = true;
				startX = e.pageX;
				startWidth = th.offsetWidth;
				resizer.classList.add('resizing');

				document.addEventListener('mousemove', onMouseMove);
				document.addEventListener('mouseup', onMouseUp);
			});

			function onMouseMove(e) {
				if (!isResizing) return;
				var newWidth = startWidth + (e.pageX - startX);
				if (newWidth > 30) {
					th.style.width = newWidth + 'px';
					th.style.minWidth = newWidth + 'px';
				}
			}

			function onMouseUp() {
				isResizing = false;
				resizer.classList.remove('resizing');
				document.removeEventListener('mousemove', onMouseMove);
				document.removeEventListener('mouseup', onMouseUp);
			}
		}

		function sortTable(table, colIndex, th) {
			var tbody = table.querySelector('tbody');
			var rows = Array.from(tbody.querySelectorAll('tr'));
			var isAsc = th.classList.contains('sort-asc');

			// Remove sort classes from all headers
			table.querySelectorAll('thead th').forEach(function(header) {
				header.classList.remove('sort-asc', 'sort-desc');
			});

			// Toggle sort direction
			if (isAsc) {
				th.classList.add('sort-desc');
			} else {
				th.classList.add('sort-asc');
			}

			var sortDirection = th.classList.contains('sort-asc') ? 1 : -1;

			rows.sort(function(a, b) {
				var aVal = a.cells[colIndex] ? a.cells[colIndex].textContent.trim() : '';
				var bVal = b.cells[colIndex] ? b.cells[colIndex].textContent.trim() : '';

				// Try numeric comparison first
				var aNum = parseFloat(aVal.replace(/,/g, ''));
				var bNum = parseFloat(bVal.replace(/,/g, ''));

				if (!isNaN(aNum) && !isNaN(bNum)) {
					return (aNum - bNum) * sortDirection;
				}

				// String comparison
				return aVal.localeCompare(bVal) * sortDirection;
			});

			// Re-append sorted rows
			rows.forEach(function(row) {
				tbody.appendChild(row);
			});
		}
	`

}


function jsFunctionVirtualScrollTable() {

	return `
	
		function responseGenerateVirtualScroll() {
			document.getElementById('nullFormatDiv').style.display = "block";
			
			if (!queryResponsePayload || !queryResponsePayload.records || queryResponsePayload.records.length === 0) {
				document.getElementById('resultsDiv').innerHTML = '<p style="color: #f59e0b; font-weight: 500;">No records found.</p>';
				return;
			}
			
			var records = queryResponsePayload.records;
			var pageSize = parseInt(document.getElementById('virtualPageSize').value) || 100;
			var totalPages = Math.ceil(records.length / pageSize);
			var columnNames = Object.keys(records[0]);
			var firstColumnIsRowNumber = document.getElementById('enablePagination').checked;
			var rowNumbersHidden = document.getElementById('hideRowNumbers') && document.getElementById('hideRowNumbers').checked;
			
			var thead = '<thead><tr>';
			for (var i = 0; i < columnNames.length; i++) {
				if (i == 0 && firstColumnIsRowNumber && rowNumbersHidden) continue;
				if (i == 0 && firstColumnIsRowNumber) {
					thead += '<th style="text-align: center; width: 50px;">#</th>';
				} else {
					thead += '<th>' + columnNames[i] + '</th>';
				}
			}
			thead += '</tr></thead>';
			
			var content = buildStatsBar();
			
			content += '<div class="pagination-controls">';
			content += '<button class="btn btn-sm btn-light" id="btnFirst" onclick="goToPage(0)">First</button>';
			content += '<button class="btn btn-sm btn-light" id="btnPrev" onclick="goToPage(window.virtualScrollData.currentPage - 1)">Prev</button>';
			content += '<span class="page-info">Page <span id="currentPageNum">1</span> / ' + totalPages + '</span>';
			content += '<button class="btn btn-sm btn-light" id="btnNext" onclick="goToPage(window.virtualScrollData.currentPage + 1)">Next</button>';
			content += '<button class="btn btn-sm btn-light" id="btnLast" onclick="goToPage(' + (totalPages - 1) + ')">Last</button>';
			content += '<input type="number" id="gotoPageInput" class="form-control-sm" style="width: 60px; margin-left: 12px;" placeholder="#" min="1" max="' + totalPages + '">';
			content += '<button class="btn btn-sm btn-primary" onclick="goToPageFromInput()">Go</button>';
			content += '</div>';
			
			content += '<div class="virtual-scroll-container" id="virtualScrollContainer">';
			content += '<table class="virtual-scroll-table" id="virtualScrollTable">';
			content += thead;
			content += '<tbody id="virtualScrollBody"></tbody>';
			content += '</table></div>';
			
			document.getElementById('resultsDiv').innerHTML = content;
			
			window.virtualScrollData = {
				records: records,
				columnNames: columnNames,
				pageSize: pageSize,
				totalPages: totalPages,
				currentPage: 0,
				firstColumnIsRowNumber: firstColumnIsRowNumber,
				rowNumbersHidden: rowNumbersHidden
			};

			renderVirtualScrollPage(0);

			// Initialize sorting and column resize for virtual scroll table
			initTableFeatures('virtualScrollTable');
		}
		
		function renderVirtualScrollPage(page) {
			var data = window.virtualScrollData;
			if (!data) return;
			
			var startIdx = page * data.pageSize;
			var endIdx = Math.min(startIdx + data.pageSize, data.records.length);
			var nullFormat = radioFieldValueGet('nullFormat');
			
			var tbody = '';
			for (var r = startIdx; r < endIdx; r++) {
				tbody += '<tr>';
				for (var i = 0; i < data.columnNames.length; i++) {
					if (i == 0 && data.firstColumnIsRowNumber && data.rowNumbersHidden) continue;
					
					var value = data.records[r][data.columnNames[i]];
					if (value === null) {
						if (nullFormat == 'dimmed') {
							value = '<span style="color: #9ca3af;">null</span>';
						} else if (nullFormat == 'blank') {
							value = '';
						} else {
							value = 'null';
						}
					} else {
						value = String(value).replace(/&/g, '&amp;').replace(/</g, '&lt;').replace(/>/g, '&gt;');
					}
					
					if (i == 0 && data.firstColumnIsRowNumber) {
						tbody += '<td style="text-align: center;">' + value + '</td>';
					} else {
						tbody += '<td>' + value + '</td>';
					}
				}
				tbody += '</tr>';
			}
			
			document.getElementById('virtualScrollBody').innerHTML = tbody;
			document.getElementById('currentPageNum').textContent = page + 1;
			
			document.getElementById('btnFirst').disabled = page === 0;
			document.getElementById('btnPrev').disabled = page === 0;
			document.getElementById('btnNext').disabled = page >= data.totalPages - 1;
			document.getElementById('btnLast').disabled = page >= data.totalPages - 1;
			
			data.currentPage = page;
			document.getElementById('virtualScrollContainer').scrollTop = 0;
		}
		
		function goToPage(page) {
			var data = window.virtualScrollData;
			if (!data) return;
			page = Math.max(0, Math.min(page, data.totalPages - 1));
			renderVirtualScrollPage(page);
		}
		
		function goToPageFromInput() {
			var input = document.getElementById('gotoPageInput');
			var page = parseInt(input.value) - 1;
			if (!isNaN(page)) goToPage(page);
			input.value = '';
		}
	`

}


function jsFunctionReturnAllToggle() {

	return `
		function returnAllToggle() {
			var rowInputs = document.getElementById('rowBegin').parentElement;
			if (document.getElementById('returnAll').checked) {
				rowInputs.style.opacity = '0.5';
				rowInputs.style.pointerEvents = 'none';
			} else {
				rowInputs.style.opacity = '1';
				rowInputs.style.pointerEvents = 'auto';
			}
		}	
	`

}


function jsFunctionTableDetailsGet() {

	return `
	
		function tableDetailsGet(tableName) {
			document.getElementById('tableInfoColumn').innerHTML = '<div class="loading-overlay"><div class="spinner"></div><p>Loading ' + tableName + '...</p></div>';
		
			var url = '/app/recordscatalog/rcendpoint.nl?action=getRecordTypeDetail&data=' + encodeURI(JSON.stringify({ scriptId: tableName, detailType: 'SS_ANAL' }));
			
			var xhr = new XMLHttpRequest();
			xhr.open('GET', url, true);
			xhr.send();
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					var recordDetail = JSON.parse(xhr.response).data;
										
					var content = '<h4 style="color: #374151; font-weight: 600; margin-bottom: 16px;">' + recordDetail.label + ' <span style="font-weight: 400; color: #6b7280;">(' + tableName + ')</span></h4>';	
					
					content += '<h5 style="font-size: 14px; font-weight: 600; margin: 16px 0 8px;">Columns</h5>';	
					content += '<table class="table table-sm table-bordered" id="tableColumnsTable">';	
					content += '<thead><tr><th>Label</th><th>Name</th><th>Type</th></tr></thead>';
					content += '<tbody>';													
					for (var i = 0; i < recordDetail.fields.length; i++) {												
						var field = recordDetail.fields[i];									
						if (field.isColumn) {
							content += '<tr><td>' + field.label + '</td><td>' + field.id + '</td><td>' + field.dataType + '</td></tr>';									
						}
					}		
					content += '</tbody></table>';

					if (recordDetail.joins.length > 0) {
						content += '<h5 style="font-size: 14px; font-weight: 600; margin: 16px 0 8px;">Joins</h5>';	
						content += '<table class="table table-sm table-bordered" id="tableJoinsTable">';
						content += '<thead><tr><th>Label</th><th>Table</th><th>Cardinality</th><th>Join Pairs</th></tr></thead>';
						content += '<tbody>';													
						for (var i = 0; i < recordDetail.joins.length; i++) {												
							var join = recordDetail.joins[i];
							var joinInfo = join.sourceTargetType.joinPairs.map(function(jp) { return jp.label; }).join(', ');
							content += '<tr>';
							content += '<td>' + join.label + '</td>';
							content += '<td><a href="#" onclick="tableDetailsGet(\\'' + join.sourceTargetType.id + '\\'); return false;">' + join.sourceTargetType.id + '</a></td>';
							content += '<td>' + join.cardinality + '</td>';
							content += '<td>' + joinInfo + '</td>';
							content += '</tr>';									
						}	
						content += '</tbody></table>';
					}	
					
					content += '<h5 style="font-size: 14px; font-weight: 600; margin: 16px 0 8px;">Sample Query</h5>';
					content += '<p style="font-size: 12px;"><a href="#" onclick="tableQueryCopy(); return false;">Copy to clipboard</a></p>';
					
					var sampleQuery = 'SELECT\\n';
					for (var i = 0; i < recordDetail.fields.length; i++) {												
						var field = recordDetail.fields[i];									
						if (field.isColumn) {
							sampleQuery += '    ' + tableName + '.' + field.id;
							if (i + 1 < recordDetail.fields.length) sampleQuery += ',';
							sampleQuery += '\\n';															
						}
					}		
					sampleQuery += 'FROM\\n    ' + tableName;
					
					content += '<textarea class="form-control" id="tableQuery" rows="' + Math.min(recordDetail.fields.length + 3, 20) + '" style="font-family: var(--font-code); font-size: 12px;">' + sampleQuery + '</textarea>';
						
					document.getElementById('tableInfoColumn').innerHTML = content;		
					
					if (${datatablesEnabled}) {
						$('#tableColumnsTable').DataTable();
						$('#tableJoinsTable').DataTable();
					}										
				} else {
					alert('Error: ' + xhr.status);
				}
			}
		}		
	`	

}


function jsFunctionTableNamesGet() {

	return `
	
		function tableNamesGet() {
			var url = '/app/recordscatalog/rcendpoint.nl?action=getRecordTypes&data=' + encodeURI(JSON.stringify({ structureType: 'FLAT' }));
			
			var xhr = new XMLHttpRequest();
			xhr.open('GET', url, true);
			xhr.send();
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					var recordTypes = JSON.parse(xhr.response).data;
					
					var content = '<table class="table table-sm table-bordered table-hover" id="tableNamesTable">';
					content += '<thead><tr><th>Table</th></tr></thead>';
					content += '<tbody>';
					for (var i = 0; i < recordTypes.length; i++) {	
						content += '<tr><td>';
						content += '<a href="#" onclick="tableDetailsGet(\\'' + recordTypes[i].id + '\\'); return false;" style="font-weight: 500;">' + recordTypes[i].label + '</a>';
						content += '<br><span style="font-size: 11px; color: #6b7280;">' + recordTypes[i].id + '</span>';
						content += '</td></tr>';													
					}	
					content += '</tbody></table>';

					document.getElementById('tablesColumn').innerHTML = content;	
					if (${datatablesEnabled}) { $('#tableNamesTable').DataTable(); }
				} else {				
					alert('Error: ' + xhr.status);
				}
			}
		}
	`
	
}


function jsFunctionTableQueryCopy() {

	return `
		function tableQueryCopy() {
			var copyText = document.getElementById("tableQuery");
			copyText.select(); 
			document.execCommand("copy");
			return false;					
		}		
	`

}


function jsFunctiontablesReferenceOpen() {

	return `
		function tablesReferenceOpen() {		
			window.open("${scriptURL}&function=tablesReference", "_tablesRef");			
		}
	`

}


function jsFunctionWorkbooksButton() {

	if ( workbooksEnabled === true ) {
		return `<button type="button" class="btn btn-sm btn-light" data-toggle="modal" data-target="#workbooksModal">Workbooks</button>`
	} else {
		return ``
	}

}


function jsFunctionWorkbookLoad() {

	return `
	
		function workbookLoad(scriptID) {
			var requestPayload = { 'function': 'workbookLoad', 'scriptID': scriptID }
			var xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', true);
			xhr.send(JSON.stringify(requestPayload));
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					var payload = JSON.parse(xhr.response);	
					
					if (payload.error == undefined) {									
						if (sqlEditor) {
							sqlEditor.setValue(payload.sql);
						} else {
							document.getElementById('query').value = payload.sql;
						}
						
						$('#workbooksModal').modal('toggle');
						document.getElementById('resultsDiv').style.display = "none";									
						
						activeSQLFile.source = 'Workbook ' + scriptID;
						activeSQLFile.fileName = '';
						activeSQLFile.description = '';
						activeSQLFile.fileID = '';
						activeSQLFile.sql = payload.sql;
						fileInfoRefresh();															
					} else {
						alert('Error: ' + payload.error);
					}																							
				} else {
					alert('Error: ' + xhr.status);								
				}
			}							
		}	
	`

}


function jsFunctionWorkbooksListGet() {

	return `
	
		function workbooksListGet() {	
			document.getElementById('workbooksList').innerHTML = '<div class="loading-overlay"><div class="spinner"></div><p>Loading workbooks...</p></div>';	
												
			var requestPayload = { 'function': 'workbooksGet' }
			var xhr = new XMLHttpRequest();
			xhr.open('POST', '${scriptURL}', true);
			xhr.send(JSON.stringify(requestPayload));
			
			xhr.onload = function() {
				if (xhr.status === 200) {	
					var payload = JSON.parse(xhr.response);
				
					if (payload.error == undefined) {	
						var content = '<table id="workbooksTable" class="table table-sm table-bordered table-hover">';
						content += '<thead><tr><th>Name</th><th>Description</th><th>Owner</th><th></th></tr></thead>';
						content += '<tbody>';
						for (var r = 0; r < payload.records.length; r++) {	
							var desc = payload.records[r].description || '';
							content += '<tr>';
							content += '<td>' + payload.records[r].name + '</td>';
							content += '<td>' + desc + '</td>';
							content += '<td>' + payload.records[r].owner + '</td>';
							content += '<td style="text-align: center;"><button type="button" class="btn btn-sm btn-primary" onclick="workbookLoad(\\'' + payload.records[r].scriptid + '\\');">Load</button></td>';
							content += '</tr>';		
						}	
						content += '</tbody></table>';
						
						document.getElementById('workbooksList').innerHTML = content;
						if (${datatablesEnabled}) { $('#workbooksTable').DataTable(); }							
					} else {
						document.getElementById('workbooksList').innerHTML = payload.error == 'No Workbooks' 
							? '<p style="color: #ef4444;">No workbooks found.</p>'
							: '<p style="color: #ef4444;">Error: ' + payload.error + '</p>';
					}																							
				} else {
					document.getElementById('workbooksList').innerHTML = '<p style="color: #ef4444;">Error: ' + xhr.status + '</p>';		
				}								
			}																	
		}	
	`

}


function localLibraryFilesGet( context ) {

	var responsePayload;

	var sql = `
		SELECT
			ID,
			Name,
			Description
		FROM
			File
		WHERE 
			( Folder = ? )
		ORDER BY 
			Name
	`;
		
	var queryResults = query.runSuiteQL( { query: sql, params: [ queryFolderID ] } ); 	

	var records = queryResults.asMappedResults();

	if ( records.length > 0 ) {
		responsePayload = { 'records': records };
	} else {
		responsePayload = { 'error': 'No SQL Files' }; 
	}	
		
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );

}


function postRequestHandle( context ) {

	var requestPayload = JSON.parse( context.request.body );
	
	context.response.setHeader( 'Content-Type', 'application/json' );
	
	switch ( requestPayload['function'] ) {
	
		case 'documentSubmit':
			return documentSubmit( context, requestPayload );
			break;				

		case 'queryExecute':
			return queryExecute( context, requestPayload );
			break;
								
		case 'sqlFileExists':
			return sqlFileExists( context, requestPayload );
			break;						
			
		case 'sqlFileLoad':
			return sqlFileLoad( context, requestPayload );
			break;			
			
		case 'sqlFileSave':
			return sqlFileSave( context, requestPayload );
			break;										
			
		case 'localLibraryFilesGet':
			return localLibraryFilesGet( context );
			break;	
			
		case 'workbookLoad':
			return workbookLoad( context, requestPayload );
			break;								
			
		case 'workbooksGet':
			return workbooksGet( context );
			break;									

		default:
			log.error( { title: 'Payload - Unsupported Function', details: requestPayload['function'] } );
		
	} 
				
}


function queryExecute( context, requestPayload ) {

	try {		
	
		var responsePayload;		
		
		var moreRecords = true;	

		var records = new Array();
		
		var totalRecordCount = 0;
		
		var queryParams = new Array();
		
		var paginatedRowBegin = requestPayload.rowBegin;
		
		var paginatedRowEnd = requestPayload.rowEnd;
		
		var nestedSQL = requestPayload.query + "\n";
		
		if ( ( requestPayload.viewsEnabled ) && ( queryFolderID !== null ) ) {
		
			var pattern = /(?:^|\s)\#(\w+)\b/ig;
			
			var views = nestedSQL.match(pattern);	
			
			if ( ( views !== null ) && ( views.length > 0 ) ) {
									
				for ( let i = 0; i < views.length; i++ ) {
			
					view = views[i].replace(/\s+/g, '');
					
					viewFileName = view.substring( 1, view.length ) + '.sql';
														
					var sql = 'SELECT ID FROM File WHERE ( Folder = ? ) AND ( Name = ? )';
		
					var queryResults = query.runSuiteQL( { query: sql, params: [ queryFolderID, viewFileName ] } ); 	

					var files = queryResults.asMappedResults();	
					
					if ( files.length == 1 ) {		
													
						var fileObj = file.load( {  id: files[0].id  } );
												
						nestedSQL = nestedSQL.replace( view, '( ' + fileObj.getContents() + ' ) AS ' + view.substring( 1, view.length ) );
																							
					} else {
					
						throw {
							'name:': 'UnresolvedViewException',
							'message': 'Unresolved View ' + viewFileName							
						}
						
					}					
								
				}
				
			}
							
		}

		let beginTime = new Date().getTime();
		
		if ( requestPayload.paginationEnabled ) {

			do {			
	
				var paginatedSQL = 'SELECT * FROM ( SELECT ROWNUM AS ROWNUMBER, * FROM ( ' + nestedSQL + ' ) ) WHERE ( ROWNUMBER BETWEEN ' + paginatedRowBegin + ' AND ' + paginatedRowEnd + ')';
		
				var queryResults = query.runSuiteQL( { query: paginatedSQL, params: queryParams } ).asMappedResults(); 	
				
				records = records.concat( queryResults );	
					
				if ( queryResults.length < 5000 ) { moreRecords = false; }
		
				paginatedRowBegin = paginatedRowBegin + 5000;
				
			} while ( moreRecords );
		
		} else {
		
			log.debug( { title: 'nestedSQL', details: nestedSQL } );
			
			records = query.runSuiteQL( { query: nestedSQL, params: queryParams } ).asMappedResults(); 	
			
			log.debug( { title: 'records', details: records } );
							
		}	
		
		let elapsedTime = ( new Date().getTime() - beginTime ) ;	
		
		responsePayload = { 'records': records, 'elapsedTime': elapsedTime }	
		
		if ( requestPayload.returnTotals ) {
		
			if ( records.length > 0 ) {
		
				var paginatedSQL = 'SELECT COUNT(*) AS TotalRecordCount FROM ( ' + nestedSQL  + ' )';
		
				var queryResults = query.runSuiteQL( { query: paginatedSQL, params: queryParams } ).asMappedResults(); 		
		
				responsePayload.totalRecordCount = queryResults[0].totalrecordcount;

			}
				
		}

	} catch( e ) {		

		log.error( { title: 'queryExecute Error', details: e } );
		
		responsePayload = { 'error': e }		
		
	}			
	
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );	
	
}


function sqlFileExists( context, requestPayload ) {
		
	var responsePayload;

	var sql = `
		SELECT
			ID
		FROM
			File
		WHERE 
			( Folder = ? ) AND ( Name = ? )
	`;
		
	var queryResults = query.runSuiteQL( { query: sql, params: [ queryFolderID, requestPayload.filename ] } ); 	

	var records = queryResults.asMappedResults();

	if ( records.length > 0 ) {
		responsePayload = { 'exists': true };
	} else {
		responsePayload = { 'exists': false }; 
	}	
		
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );	

}


function sqlFileLoad( context, requestPayload ) {

	var responsePayload;
	
	try {

		var fileObj = file.load( {  id: requestPayload.fileID  } );
						
		responsePayload = {}
		responsePayload.file = fileObj;
		responsePayload.sql = fileObj.getContents();				
		
	} catch( e ) {		

		log.error( { title: 'sqlFileLoad Error', details: e } );
		
		responsePayload = { 'error': e }		
		
	}	
		
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );			

}


function sqlFileSave( context, requestPayload ) {

	var responsePayload;
	
	try {

		var fileObj = file.create( 
			{
				name: requestPayload.filename,
				contents: requestPayload.contents,
				description: requestPayload.description,
				fileType: file.Type.PLAINTEXT,
				folder: queryFolderID,
				isOnline: false
			} 
		);
	
		var fileID = fileObj.save();
						
		responsePayload = {}
		responsePayload.fileID = fileID;
		
	} catch( e ) {		

		log.error( { title: 'sqlFileSave Error', details: e } );
		
		responsePayload = { 'error': e }		
		
	}	
		
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );			

}


function workbookLoad( context, requestPayload ) {

	var responsePayload;
	
	try {
	
		var loadedQuery = query.load( { id: requestPayload.scriptID } );	
    								
		responsePayload = {}
		responsePayload.sql = loadedQuery.toSuiteQL().query;				
		
	} catch( e ) {		

		log.error( { title: 'workbookLoad Error', details: e } );
		
		responsePayload = { 'error': e }		
		
	}	
		
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );		

}


function workbooksGet( context ) {

	var responsePayload;

	var sql = `
		SELECT
			ScriptID,
			Name,
			Description,
			BUILTIN.DF( Owner ) AS Owner
		FROM
			UsrSavedSearch
		ORDER BY
			Name
	`;
		
	var queryResults = query.runSuiteQL( { query: sql, params: [] } ); 	

	var records = queryResults.asMappedResults();

	if ( records.length > 0 ) {
		responsePayload = { 'records': records };
	} else {
		responsePayload = { 'error': 'No Workbooks' }; 
	}	
		
	context.response.write( JSON.stringify( responsePayload, null, 5 ) );

}
